/*
 * $Id: unix_chkpwd.c,v 1.1 1996/11/10 21:20:51 morgan Exp $
 *
 * This program is designed to run setuid(root) or with sufficient
 * privilege to read all of the unix password databases. It is designed
 * to provide a mechanism for the current user (defined by this
 * process' uid) to verify their own password.
 *
 * The password is read from the standard input. The exit status of
 * this program indicates whether the user is authenticated or not.
 *
 * Copyright information is located at the end of the file.
 *
 * $Log: unix_chkpwd.c,v $
 * Revision 1.1  1996/11/10 21:20:51  morgan
 * Initial revision
 *
 */

#define MAXPASS      200        /* the maximum length of a password */

#define UNIX_CORRECT 0
#define UNIX_FAILED  1

#include <security/pwdb.h>

#include "pam_md.-c"

static int _unix_verify_password(const char *name, const char *p)
{
     struct __pwdb_passwd *pw;
     struct __pwdb_spwd *sp;
     const char *salt;
     char *pp;
     int retval;

     pw = __pwdb_getpwnam(name);
     if (pw) {                           /* does user exist? */

	  if ( strcmp(pw->pw_passwd, __PWDB_SHADOW_TOKEN) == 0 ) {
	       /*
		* Linux-PAM is pluggable, so the sys-admin may have
		* some users authenticated against shadow entries and
		* some without. An "x" in the user's password field of
		* the /etc/passwd file indicates the use of
		* /etc/shadow is manditory for that user. An 'x' will be
		* placed there by the chauthtok functions if 'shadow' is
		* supplied as an argument of the module.
		*/

	       sp = __pwdb_getspnam( name );
	       if (sp) {
		    salt = sp->sp_pwdp;
	       } else {
		    _log_err(LOG_ALERT, "user %s incorrectly shadowed", name);
		    p = NULL;
		    return UNIX_FAIL;
	       }
	  } else {
	       salt = pw->pw_passwd;
	  }
     } else {
	  salt = "xx";
     }

     pp = _pam_md(p, salt);
     p = NULL;                     /* no longer needed here */

     if ( strcmp( pp, salt ) == 0 ) {
	  retval = UNIX_PASSED;
     } else {
	  retval = UNIX_FAILED;
     }

     /* clean up */
     {
	  char *tp = pp;
	  if (pp != NULL) {
	       while(tp && *tp)
		    *tp++ = '\0';
	       free(pp);
	  }
	  pp = tp = NULL;
     }

     return retval;
}

int main()
{
     char pass[MAXPASS+1];
     int npass;
     int retval = UNIX_FAILED;

     /*
      * determine the current user's name is
      * 1. supplied as a environment variable as LOGNAME
      * 2. the uid has to match the one associated with the LOGNAME.
      */

     user = getuidname(getuid());

     /* read the password from stdin (a pipe from the pam_unix module) */

     npass = read(STDIN_FILENO, pass, MAXPASS);

     /*
      * we establish that this program is running with non-tty stdin.
      * this is to discourage casual use. It does *NOT* prevent an
      * intruder from repeatadly running this program to determine the
      * password of the current user (brute force attack, but one for
      * which the attacker must already have gained access to the user's
      * account).
      */

     if ( isatty(STDIN_FILENO) == ?? ) {

	  _log_err(LOG_NOTICE
		   , "inappropriate use of Unix helper binary [UID=%d]"
		   , getuid() );
	  fprintf(stderr
		  , "This binary is not designed for running in this way\n"
		    "-- the system administrator has been informed\n");
	  sleep(10);              /* this should discourage/annoy the user */

     } else if (npass < 0) {                   /* is it a valid password? */

	  _log_err(LOG_DEBUG, "no password supplied");

     } else if (npass >= MAXPASS) {

	  _log_err(LOG_DEBUG, "password too long");

     } else {
	  if (npass == 0) {
	       /* the password is NULL */

	       retval = _unix_verify_passwd(user, NULL);

	  } else {
	       /* does pass agree with the official one? */

	       pass[npass] = '\0';                     /* NUL terminate */
	       retval = _unix_verify_passwd(user,pass);

	  }
     }

     memset(pass, '\0', MAXPASS);        /* clear memory of the password */

     /* return pass or fail */

     return retval;
}

/*
 * Copyright (c) Andrew G. Morgan, 1996. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 * 
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
