/*
 *   fs/smb2/smb2maperror.c
 *
 *   Functions which do error mapping of SMB2 status codes to POSIX errors
 *
 *   Copyright (C) International Business Machines  Corp., 2009
 *   Author(s): Steve French (sfrench@us.ibm.com)
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation; either version 2.1 of the License, or
 *   (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/errno.h>
#include "cifsglob.h"
#include "cifs_debug.h"
#include "smb2pdu.h"
#include "smb2proto.h"
#include "smb2status.h"
#include "smb2glob.h"

struct status_to_posix_error {
	__le32 smb2_status;
	int posix_error;
	char *status_string;
};

static const struct status_to_posix_error smb2_error_map_table[] = {
	{STATUS_SUCCESS, 0, "STATUS_SUCCESS"},
	{STATUS_WAIT_0,  0, "STATUS_WAIT_0"},
	{STATUS_WAIT_1, -EIO, "STATUS_WAIT_1"},
	{STATUS_WAIT_2, -EIO, "STATUS_WAIT_2"},
	{STATUS_WAIT_3, -EIO, "STATUS_WAIT_3"},
	{STATUS_WAIT_63, -EIO, "STATUS_WAIT_63"},
	{STATUS_ABANDONED, -EIO, "STATUS_ABANDONED"},
	{STATUS_ABANDONED_WAIT_0, -EIO, "STATUS_ABANDONED_WAIT_0"},
	{STATUS_ABANDONED_WAIT_63, -EIO, "STATUS_ABANDONED_WAIT_63"},
	{STATUS_USER_APC, -EIO, "STATUS_USER_APC"},
	{STATUS_KERNEL_APC, -EIO, "STATUS_KERNEL_APC"},
	{STATUS_ALERTED, -EIO, "STATUS_ALERTED"},
	{STATUS_TIMEOUT, -ETIMEDOUT, "STATUS_TIMEOUT"},
	{STATUS_PENDING, -EIO, "STATUS_PENDING"},
	{STATUS_REPARSE, -EIO, "STATUS_REPARSE"},
	{STATUS_MORE_ENTRIES, -EIO, "STATUS_MORE_ENTRIES"},
	{STATUS_NOT_ALL_ASSIGNED, -EIO, "STATUS_NOT_ALL_ASSIGNED"},
	{STATUS_SOME_NOT_MAPPED, -EIO, "STATUS_SOME_NOT_MAPPED"},
	{STATUS_OPLOCK_BREAK_IN_PROGRESS, -EIO,
	"STATUS_OPLOCK_BREAK_IN_PROGRESS"},
	{STATUS_VOLUME_MOUNTED, -EIO, "STATUS_VOLUME_MOUNTED"},
	{STATUS_RXACT_COMMITTED, -EIO, "STATUS_RXACT_COMMITTED"},
	{STATUS_NOTIFY_CLEANUP, -EIO, "STATUS_NOTIFY_CLEANUP"},
	{STATUS_NOTIFY_ENUM_DIR, -EIO, "STATUS_NOTIFY_ENUM_DIR"},
	{STATUS_NO_QUOTAS_FOR_ACCOUNT, -EIO, "STATUS_NO_QUOTAS_FOR_ACCOUNT"},
	{STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED, -EIO,
	"STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED"},
	{STATUS_PAGE_FAULT_TRANSITION, -EIO, "STATUS_PAGE_FAULT_TRANSITION"},
	{STATUS_PAGE_FAULT_DEMAND_ZERO, -EIO, "STATUS_PAGE_FAULT_DEMAND_ZERO"},
	{STATUS_PAGE_FAULT_COPY_ON_WRITE, -EIO,
	"STATUS_PAGE_FAULT_COPY_ON_WRITE"},
	{STATUS_PAGE_FAULT_GUARD_PAGE, -EIO, "STATUS_PAGE_FAULT_GUARD_PAGE"},
	{STATUS_PAGE_FAULT_PAGING_FILE, -EIO, "STATUS_PAGE_FAULT_PAGING_FILE"},
	{STATUS_CACHE_PAGE_LOCKED, -EIO, "STATUS_CACHE_PAGE_LOCKED"},
	{STATUS_CRASH_DUMP, -EIO, "STATUS_CRASH_DUMP"},
	{STATUS_BUFFER_ALL_ZEROS, -EIO, "STATUS_BUFFER_ALL_ZEROS"},
	{STATUS_REPARSE_OBJECT, -EIO, "STATUS_REPARSE_OBJECT"},
	{STATUS_RESOURCE_REQUIREMENTS_CHANGED, -EIO,
	"STATUS_RESOURCE_REQUIREMENTS_CHANGED"},
	{STATUS_TRANSLATION_COMPLETE, -EIO, "STATUS_TRANSLATION_COMPLETE"},
	{STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY, -EIO,
	"STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY"},
	{STATUS_NOTHING_TO_TERMINATE, -EIO, "STATUS_NOTHING_TO_TERMINATE"},
	{STATUS_PROCESS_NOT_IN_JOB, -EIO, "STATUS_PROCESS_NOT_IN_JOB"},
	{STATUS_PROCESS_IN_JOB, -EIO, "STATUS_PROCESS_IN_JOB"},
	{STATUS_VOLSNAP_HIBERNATE_READY, -EIO,
	"STATUS_VOLSNAP_HIBERNATE_READY"},
	{STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY, -EIO,
	"STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY"},
	{STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED, -EIO,
	"STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED"},
	{STATUS_INTERRUPT_STILL_CONNECTED, -EIO,
	"STATUS_INTERRUPT_STILL_CONNECTED"},
	{STATUS_PROCESS_CLONED, -EIO, "STATUS_PROCESS_CLONED"},
	{STATUS_FILE_LOCKED_WITH_ONLY_READERS, -EIO,
	"STATUS_FILE_LOCKED_WITH_ONLY_READERS"},
	{STATUS_FILE_LOCKED_WITH_WRITERS, -EIO,
	"STATUS_FILE_LOCKED_WITH_WRITERS"},
	{STATUS_RESOURCEMANAGER_READ_ONLY, -EROFS,
	"STATUS_RESOURCEMANAGER_READ_ONLY"},
	{STATUS_WAIT_FOR_OPLOCK, -EIO, "STATUS_WAIT_FOR_OPLOCK"},
	{DBG_EXCEPTION_HANDLED, -EIO, "DBG_EXCEPTION_HANDLED"},
	{DBG_CONTINUE, -EIO, "DBG_CONTINUE"},
	{STATUS_FLT_IO_COMPLETE, -EIO, "STATUS_FLT_IO_COMPLETE"},
	{STATUS_OBJECT_NAME_EXISTS, -EIO, "STATUS_OBJECT_NAME_EXISTS"},
	{STATUS_THREAD_WAS_SUSPENDED, -EIO, "STATUS_THREAD_WAS_SUSPENDED"},
	{STATUS_WORKING_SET_LIMIT_RANGE, -EIO,
	"STATUS_WORKING_SET_LIMIT_RANGE"},
	{STATUS_IMAGE_NOT_AT_BASE, -EIO, "STATUS_IMAGE_NOT_AT_BASE"},
	{STATUS_RXACT_STATE_CREATED, -EIO, "STATUS_RXACT_STATE_CREATED"},
	{STATUS_SEGMENT_NOTIFICATION, -EIO, "STATUS_SEGMENT_NOTIFICATION"},
	{STATUS_LOCAL_USER_SESSION_KEY, -EIO, "STATUS_LOCAL_USER_SESSION_KEY"},
	{STATUS_BAD_CURRENT_DIRECTORY, -EIO, "STATUS_BAD_CURRENT_DIRECTORY"},
	{STATUS_SERIAL_MORE_WRITES, -EIO, "STATUS_SERIAL_MORE_WRITES"},
	{STATUS_REGISTRY_RECOVERED, -EIO, "STATUS_REGISTRY_RECOVERED"},
	{STATUS_FT_READ_RECOVERY_FROM_BACKUP, -EIO,
	"STATUS_FT_READ_RECOVERY_FROM_BACKUP"},
	{STATUS_FT_WRITE_RECOVERY, -EIO, "STATUS_FT_WRITE_RECOVERY"},
	{STATUS_SERIAL_COUNTER_TIMEOUT, -ETIMEDOUT,
	"STATUS_SERIAL_COUNTER_TIMEOUT"},
	{STATUS_NULL_LM_PASSWORD, -EIO, "STATUS_NULL_LM_PASSWORD"},
	{STATUS_IMAGE_MACHINE_TYPE_MISMATCH, -EIO,
	"STATUS_IMAGE_MACHINE_TYPE_MISMATCH"},
	{STATUS_RECEIVE_PARTIAL, -EIO, "STATUS_RECEIVE_PARTIAL"},
	{STATUS_RECEIVE_EXPEDITED, -EIO, "STATUS_RECEIVE_EXPEDITED"},
	{STATUS_RECEIVE_PARTIAL_EXPEDITED, -EIO,
	"STATUS_RECEIVE_PARTIAL_EXPEDITED"},
	{STATUS_EVENT_DONE, -EIO, "STATUS_EVENT_DONE"},
	{STATUS_EVENT_PENDING, -EIO, "STATUS_EVENT_PENDING"},
	{STATUS_CHECKING_FILE_SYSTEM, -EIO, "STATUS_CHECKING_FILE_SYSTEM"},
	{STATUS_FATAL_APP_EXIT, -EIO, "STATUS_FATAL_APP_EXIT"},
	{STATUS_PREDEFINED_HANDLE, -EIO, "STATUS_PREDEFINED_HANDLE"},
	{STATUS_WAS_UNLOCKED, -EIO, "STATUS_WAS_UNLOCKED"},
	{STATUS_SERVICE_NOTIFICATION, -EIO, "STATUS_SERVICE_NOTIFICATION"},
	{STATUS_WAS_LOCKED, -EIO, "STATUS_WAS_LOCKED"},
	{STATUS_LOG_HARD_ERROR, -EIO, "STATUS_LOG_HARD_ERROR"},
	{STATUS_ALREADY_WIN32, -EIO, "STATUS_ALREADY_WIN32"},
	{STATUS_WX86_UNSIMULATE, -EIO, "STATUS_WX86_UNSIMULATE"},
	{STATUS_WX86_CONTINUE, -EIO, "STATUS_WX86_CONTINUE"},
	{STATUS_WX86_SINGLE_STEP, -EIO, "STATUS_WX86_SINGLE_STEP"},
	{STATUS_WX86_BREAKPOINT, -EIO, "STATUS_WX86_BREAKPOINT"},
	{STATUS_WX86_EXCEPTION_CONTINUE, -EIO,
	"STATUS_WX86_EXCEPTION_CONTINUE"},
	{STATUS_WX86_EXCEPTION_LASTCHANCE, -EIO,
	"STATUS_WX86_EXCEPTION_LASTCHANCE"},
	{STATUS_WX86_EXCEPTION_CHAIN, -EIO, "STATUS_WX86_EXCEPTION_CHAIN"},
	{STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE, -EIO,
	"STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE"},
	{STATUS_NO_YIELD_PERFORMED, -EIO, "STATUS_NO_YIELD_PERFORMED"},
	{STATUS_TIMER_RESUME_IGNORED, -EIO, "STATUS_TIMER_RESUME_IGNORED"},
	{STATUS_ARBITRATION_UNHANDLED, -EIO, "STATUS_ARBITRATION_UNHANDLED"},
	{STATUS_CARDBUS_NOT_SUPPORTED, -ENOSYS, "STATUS_CARDBUS_NOT_SUPPORTED"},
	{STATUS_WX86_CREATEWX86TIB, -EIO, "STATUS_WX86_CREATEWX86TIB"},
	{STATUS_MP_PROCESSOR_MISMATCH, -EIO, "STATUS_MP_PROCESSOR_MISMATCH"},
	{STATUS_HIBERNATED, -EIO, "STATUS_HIBERNATED"},
	{STATUS_RESUME_HIBERNATION, -EIO, "STATUS_RESUME_HIBERNATION"},
	{STATUS_FIRMWARE_UPDATED, -EIO, "STATUS_FIRMWARE_UPDATED"},
	{STATUS_DRIVERS_LEAKING_LOCKED_PAGES, -EIO,
	"STATUS_DRIVERS_LEAKING_LOCKED_PAGES"},
	{STATUS_MESSAGE_RETRIEVED, -EIO, "STATUS_MESSAGE_RETRIEVED"},
	{STATUS_SYSTEM_POWERSTATE_TRANSITION, -EIO,
	"STATUS_SYSTEM_POWERSTATE_TRANSITION"},
	{STATUS_ALPC_CHECK_COMPLETION_LIST, -EIO,
	"STATUS_ALPC_CHECK_COMPLETION_LIST"},
	{STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION, -EIO,
	"STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION"},
	{STATUS_ACCESS_AUDIT_BY_POLICY, -EIO, "STATUS_ACCESS_AUDIT_BY_POLICY"},
	{STATUS_ABANDON_HIBERFILE, -EIO, "STATUS_ABANDON_HIBERFILE"},
	{STATUS_BIZRULES_NOT_ENABLED, -EIO, "STATUS_BIZRULES_NOT_ENABLED"},
	{STATUS_WAKE_SYSTEM, -EIO, "STATUS_WAKE_SYSTEM"},
	{STATUS_DS_SHUTTING_DOWN, -EIO, "STATUS_DS_SHUTTING_DOWN"},
	{DBG_REPLY_LATER, -EIO, "DBG_REPLY_LATER"},
	{DBG_UNABLE_TO_PROVIDE_HANDLE, -EIO, "DBG_UNABLE_TO_PROVIDE_HANDLE"},
	{DBG_TERMINATE_THREAD, -EIO, "DBG_TERMINATE_THREAD"},
	{DBG_TERMINATE_PROCESS, -EIO, "DBG_TERMINATE_PROCESS"},
	{DBG_CONTROL_C, -EIO, "DBG_CONTROL_C"},
	{DBG_PRINTEXCEPTION_C, -EIO, "DBG_PRINTEXCEPTION_C"},
	{DBG_RIPEXCEPTION, -EIO, "DBG_RIPEXCEPTION"},
	{DBG_CONTROL_BREAK, -EIO, "DBG_CONTROL_BREAK"},
	{DBG_COMMAND_EXCEPTION, -EIO, "DBG_COMMAND_EXCEPTION"},
	{RPC_NT_UUID_LOCAL_ONLY, -EIO, "RPC_NT_UUID_LOCAL_ONLY"},
	{RPC_NT_SEND_INCOMPLETE, -EIO, "RPC_NT_SEND_INCOMPLETE"},
	{STATUS_CTX_CDM_CONNECT, -EIO, "STATUS_CTX_CDM_CONNECT"},
	{STATUS_CTX_CDM_DISCONNECT, -EIO, "STATUS_CTX_CDM_DISCONNECT"},
	{STATUS_SXS_RELEASE_ACTIVATION_CONTEXT, -EIO,
	"STATUS_SXS_RELEASE_ACTIVATION_CONTEXT"},
	{STATUS_RECOVERY_NOT_NEEDED, -EIO, "STATUS_RECOVERY_NOT_NEEDED"},
	{STATUS_RM_ALREADY_STARTED, -EIO, "STATUS_RM_ALREADY_STARTED"},
	{STATUS_LOG_NO_RESTART, -EIO, "STATUS_LOG_NO_RESTART"},
	{STATUS_VIDEO_DRIVER_DEBUG_REPORT_REQUEST, -EIO,
	"STATUS_VIDEO_DRIVER_DEBUG_REPORT_REQUEST"},
	{STATUS_GRAPHICS_PARTIAL_DATA_POPULATED, -EIO,
	"STATUS_GRAPHICS_PARTIAL_DATA_POPULATED"},
	{STATUS_GRAPHICS_DRIVER_MISMATCH, -EIO,
	"STATUS_GRAPHICS_DRIVER_MISMATCH"},
	{STATUS_GRAPHICS_MODE_NOT_PINNED, -EIO,
	"STATUS_GRAPHICS_MODE_NOT_PINNED"},
	{STATUS_GRAPHICS_NO_PREFERRED_MODE, -EIO,
	"STATUS_GRAPHICS_NO_PREFERRED_MODE"},
	{STATUS_GRAPHICS_DATASET_IS_EMPTY, -EIO,
	"STATUS_GRAPHICS_DATASET_IS_EMPTY"},
	{STATUS_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET, -EIO,
	"STATUS_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET"},
	{STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED, -EIO,
	"STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED"},
	{STATUS_GRAPHICS_UNKNOWN_CHILD_STATUS, -EIO,
	"STATUS_GRAPHICS_UNKNOWN_CHILD_STATUS"},
	{STATUS_GRAPHICS_LEADLINK_START_DEFERRED, -EIO,
	"STATUS_GRAPHICS_LEADLINK_START_DEFERRED"},
	{STATUS_GRAPHICS_POLLING_TOO_FREQUENTLY, -EIO,
	"STATUS_GRAPHICS_POLLING_TOO_FREQUENTLY"},
	{STATUS_GRAPHICS_START_DEFERRED, -EIO,
	"STATUS_GRAPHICS_START_DEFERRED"},
	{STATUS_NDIS_INDICATION_REQUIRED, -EIO,
	"STATUS_NDIS_INDICATION_REQUIRED"},
	{STATUS_GUARD_PAGE_VIOLATION, -EIO, "STATUS_GUARD_PAGE_VIOLATION"},
	{STATUS_DATATYPE_MISALIGNMENT, -EIO, "STATUS_DATATYPE_MISALIGNMENT"},
	{STATUS_BREAKPOINT, -EIO, "STATUS_BREAKPOINT"},
	{STATUS_SINGLE_STEP, -EIO, "STATUS_SINGLE_STEP"},
	{STATUS_BUFFER_OVERFLOW, -E2BIG, "STATUS_BUFFER_OVERFLOW"},
	{STATUS_NO_MORE_FILES, -ENODATA, "STATUS_NO_MORE_FILES"},
	{STATUS_WAKE_SYSTEM_DEBUGGER, -EIO, "STATUS_WAKE_SYSTEM_DEBUGGER"},
	{STATUS_HANDLES_CLOSED, -EIO, "STATUS_HANDLES_CLOSED"},
	{STATUS_NO_INHERITANCE, -EIO, "STATUS_NO_INHERITANCE"},
	{STATUS_GUID_SUBSTITUTION_MADE, -EIO, "STATUS_GUID_SUBSTITUTION_MADE"},
	{STATUS_PARTIAL_COPY, -EIO, "STATUS_PARTIAL_COPY"},
	{STATUS_DEVICE_PAPER_EMPTY, -EIO, "STATUS_DEVICE_PAPER_EMPTY"},
	{STATUS_DEVICE_POWERED_OFF, -EIO, "STATUS_DEVICE_POWERED_OFF"},
	{STATUS_DEVICE_OFF_LINE, -EIO, "STATUS_DEVICE_OFF_LINE"},
	{STATUS_DEVICE_BUSY, -EBUSY, "STATUS_DEVICE_BUSY"},
	{STATUS_NO_MORE_EAS, -EIO, "STATUS_NO_MORE_EAS"},
	{STATUS_INVALID_EA_NAME, -EINVAL, "STATUS_INVALID_EA_NAME"},
	{STATUS_EA_LIST_INCONSISTENT, -EIO, "STATUS_EA_LIST_INCONSISTENT"},
	{STATUS_INVALID_EA_FLAG, -EINVAL, "STATUS_INVALID_EA_FLAG"},
	{STATUS_VERIFY_REQUIRED, -EIO, "STATUS_VERIFY_REQUIRED"},
	{STATUS_EXTRANEOUS_INFORMATION, -EIO, "STATUS_EXTRANEOUS_INFORMATION"},
	{STATUS_RXACT_COMMIT_NECESSARY, -EIO, "STATUS_RXACT_COMMIT_NECESSARY"},
	{STATUS_NO_MORE_ENTRIES, -EIO, "STATUS_NO_MORE_ENTRIES"},
	{STATUS_FILEMARK_DETECTED, -EIO, "STATUS_FILEMARK_DETECTED"},
	{STATUS_MEDIA_CHANGED, -EIO, "STATUS_MEDIA_CHANGED"},
	{STATUS_BUS_RESET, -EIO, "STATUS_BUS_RESET"},
	{STATUS_END_OF_MEDIA, -EIO, "STATUS_END_OF_MEDIA"},
	{STATUS_BEGINNING_OF_MEDIA, -EIO, "STATUS_BEGINNING_OF_MEDIA"},
	{STATUS_MEDIA_CHECK, -EIO, "STATUS_MEDIA_CHECK"},
	{STATUS_SETMARK_DETECTED, -EIO, "STATUS_SETMARK_DETECTED"},
	{STATUS_NO_DATA_DETECTED, -EIO, "STATUS_NO_DATA_DETECTED"},
	{STATUS_REDIRECTOR_HAS_OPEN_HANDLES, -EIO,
	"STATUS_REDIRECTOR_HAS_OPEN_HANDLES"},
	{STATUS_SERVER_HAS_OPEN_HANDLES, -EIO,
	"STATUS_SERVER_HAS_OPEN_HANDLES"},
	{STATUS_ALREADY_DISCONNECTED, -EIO, "STATUS_ALREADY_DISCONNECTED"},
	{STATUS_LONGJUMP, -EIO, "STATUS_LONGJUMP"},
	{STATUS_CLEANER_CARTRIDGE_INSTALLED, -EIO,
	"STATUS_CLEANER_CARTRIDGE_INSTALLED"},
	{STATUS_PLUGPLAY_QUERY_VETOED, -EIO, "STATUS_PLUGPLAY_QUERY_VETOED"},
	{STATUS_UNWIND_CONSOLIDATE, -EIO, "STATUS_UNWIND_CONSOLIDATE"},
	{STATUS_REGISTRY_HIVE_RECOVERED, -EIO,
	"STATUS_REGISTRY_HIVE_RECOVERED"},
	{STATUS_DLL_MIGHT_BE_INSECURE, -EIO, "STATUS_DLL_MIGHT_BE_INSECURE"},
	{STATUS_DLL_MIGHT_BE_INCOMPATIBLE, -EIO,
	"STATUS_DLL_MIGHT_BE_INCOMPATIBLE"},
	{STATUS_STOPPED_ON_SYMLINK, -EOPNOTSUPP, "STATUS_STOPPED_ON_SYMLINK"},
	{STATUS_IO_REPARSE_TAG_NOT_HANDLED, -EOPNOTSUPP,
	"STATUS_REPARSE_NOT_HANDLED"},
	{STATUS_DEVICE_REQUIRES_CLEANING, -EIO,
	"STATUS_DEVICE_REQUIRES_CLEANING"},
	{STATUS_DEVICE_DOOR_OPEN, -EIO, "STATUS_DEVICE_DOOR_OPEN"},
	{STATUS_DATA_LOST_REPAIR, -EIO, "STATUS_DATA_LOST_REPAIR"},
	{DBG_EXCEPTION_NOT_HANDLED, -EIO, "DBG_EXCEPTION_NOT_HANDLED"},
	{STATUS_CLUSTER_NODE_ALREADY_UP, -EIO,
	"STATUS_CLUSTER_NODE_ALREADY_UP"},
	{STATUS_CLUSTER_NODE_ALREADY_DOWN, -EIO,
	"STATUS_CLUSTER_NODE_ALREADY_DOWN"},
	{STATUS_CLUSTER_NETWORK_ALREADY_ONLINE, -EIO,
	"STATUS_CLUSTER_NETWORK_ALREADY_ONLINE"},
	{STATUS_CLUSTER_NETWORK_ALREADY_OFFLINE, -EIO,
	"STATUS_CLUSTER_NETWORK_ALREADY_OFFLINE"},
	{STATUS_CLUSTER_NODE_ALREADY_MEMBER, -EIO,
	"STATUS_CLUSTER_NODE_ALREADY_MEMBER"},
	{STATUS_COULD_NOT_RESIZE_LOG, -EIO, "STATUS_COULD_NOT_RESIZE_LOG"},
	{STATUS_NO_TXF_METADATA, -EIO, "STATUS_NO_TXF_METADATA"},
	{STATUS_CANT_RECOVER_WITH_HANDLE_OPEN, -EIO,
	"STATUS_CANT_RECOVER_WITH_HANDLE_OPEN"},
	{STATUS_TXF_METADATA_ALREADY_PRESENT, -EIO,
	"STATUS_TXF_METADATA_ALREADY_PRESENT"},
	{STATUS_TRANSACTION_SCOPE_CALLBACKS_NOT_SET, -EIO,
	"STATUS_TRANSACTION_SCOPE_CALLBACKS_NOT_SET"},
	{STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD_RECOVERED, -EIO,
	"STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD_RECOVERED"},
	{STATUS_FLT_BUFFER_TOO_SMALL, -ENOBUFS, "STATUS_FLT_BUFFER_TOO_SMALL"},
	{STATUS_FVE_PARTIAL_METADATA, -EIO, "STATUS_FVE_PARTIAL_METADATA"},
	{STATUS_UNSUCCESSFUL, -EIO, "STATUS_UNSUCCESSFUL"},
	{STATUS_NOT_IMPLEMENTED, -ENOSYS, "STATUS_NOT_IMPLEMENTED"},
	{STATUS_INVALID_INFO_CLASS, -EIO, "STATUS_INVALID_INFO_CLASS"},
	{STATUS_INFO_LENGTH_MISMATCH, -EIO, "STATUS_INFO_LENGTH_MISMATCH"},
	{STATUS_ACCESS_VIOLATION, -EACCES, "STATUS_ACCESS_VIOLATION"},
	{STATUS_IN_PAGE_ERROR, -EFAULT, "STATUS_IN_PAGE_ERROR"},
	{STATUS_PAGEFILE_QUOTA, -EDQUOT, "STATUS_PAGEFILE_QUOTA"},
	{STATUS_INVALID_HANDLE, -EBADF, "STATUS_INVALID_HANDLE"},
	{STATUS_BAD_INITIAL_STACK, -EIO, "STATUS_BAD_INITIAL_STACK"},
	{STATUS_BAD_INITIAL_PC, -EIO, "STATUS_BAD_INITIAL_PC"},
	{STATUS_INVALID_CID, -EIO, "STATUS_INVALID_CID"},
	{STATUS_TIMER_NOT_CANCELED, -EIO, "STATUS_TIMER_NOT_CANCELED"},
	{STATUS_INVALID_PARAMETER, -EINVAL, "STATUS_INVALID_PARAMETER"},
	{STATUS_NO_SUCH_DEVICE, -ENODEV, "STATUS_NO_SUCH_DEVICE"},
	{STATUS_NO_SUCH_FILE, -ENOENT, "STATUS_NO_SUCH_FILE"},
	{STATUS_INVALID_DEVICE_REQUEST, -EOPNOTSUPP, "STATUS_INVALID_DEVICE_REQUEST"},
	{STATUS_END_OF_FILE, -ENODATA, "STATUS_END_OF_FILE"},
	{STATUS_WRONG_VOLUME, -EIO, "STATUS_WRONG_VOLUME"},
	{STATUS_NO_MEDIA_IN_DEVICE, -EIO, "STATUS_NO_MEDIA_IN_DEVICE"},
	{STATUS_UNRECOGNIZED_MEDIA, -EIO, "STATUS_UNRECOGNIZED_MEDIA"},
	{STATUS_NONEXISTENT_SECTOR, -EIO, "STATUS_NONEXISTENT_SECTOR"},
	{STATUS_MORE_PROCESSING_REQUIRED, -EIO,
	"STATUS_MORE_PROCESSING_REQUIRED"},
	{STATUS_NO_MEMORY, -EREMOTEIO, "STATUS_NO_MEMORY"},
	{STATUS_CONFLICTING_ADDRESSES, -EADDRINUSE,
	"STATUS_CONFLICTING_ADDRESSES"},
	{STATUS_NOT_MAPPED_VIEW, -EIO, "STATUS_NOT_MAPPED_VIEW"},
	{STATUS_UNABLE_TO_FREE_VM, -EIO, "STATUS_UNABLE_TO_FREE_VM"},
	{STATUS_UNABLE_TO_DELETE_SECTION, -EIO,
	"STATUS_UNABLE_TO_DELETE_SECTION"},
	{STATUS_INVALID_SYSTEM_SERVICE, -EIO, "STATUS_INVALID_SYSTEM_SERVICE"},
	{STATUS_ILLEGAL_INSTRUCTION, -EIO, "STATUS_ILLEGAL_INSTRUCTION"},
	{STATUS_INVALID_LOCK_SEQUENCE, -EIO, "STATUS_INVALID_LOCK_SEQUENCE"},
	{STATUS_INVALID_VIEW_SIZE, -EIO, "STATUS_INVALID_VIEW_SIZE"},
	{STATUS_INVALID_FILE_FOR_SECTION, -EIO,
	"STATUS_INVALID_FILE_FOR_SECTION"},
	{STATUS_ALREADY_COMMITTED, -EIO, "STATUS_ALREADY_COMMITTED"},
	{STATUS_ACCESS_DENIED, -EACCES, "STATUS_ACCESS_DENIED"},
	{STATUS_BUFFER_TOO_SMALL, -EIO, "STATUS_BUFFER_TOO_SMALL"},
	{STATUS_OBJECT_TYPE_MISMATCH, -EIO, "STATUS_OBJECT_TYPE_MISMATCH"},
	{STATUS_NONCONTINUABLE_EXCEPTION, -EIO,
	"STATUS_NONCONTINUABLE_EXCEPTION"},
	{STATUS_INVALID_DISPOSITION, -EIO, "STATUS_INVALID_DISPOSITION"},
	{STATUS_UNWIND, -EIO, "STATUS_UNWIND"},
	{STATUS_BAD_STACK, -EIO, "STATUS_BAD_STACK"},
	{STATUS_INVALID_UNWIND_TARGET, -EIO, "STATUS_INVALID_UNWIND_TARGET"},
	{STATUS_NOT_LOCKED, -EIO, "STATUS_NOT_LOCKED"},
	{STATUS_PARITY_ERROR, -EIO, "STATUS_PARITY_ERROR"},
	{STATUS_UNABLE_TO_DECOMMIT_VM, -EIO, "STATUS_UNABLE_TO_DECOMMIT_VM"},
	{STATUS_NOT_COMMITTED, -EIO, "STATUS_NOT_COMMITTED"},
	{STATUS_INVALID_PORT_ATTRIBUTES, -EIO,
	"STATUS_INVALID_PORT_ATTRIBUTES"},
	{STATUS_PORT_MESSAGE_TOO_LONG, -EIO, "STATUS_PORT_MESSAGE_TOO_LONG"},
	{STATUS_INVALID_PARAMETER_MIX, -EINVAL, "STATUS_INVALID_PARAMETER_MIX"},
	{STATUS_INVALID_QUOTA_LOWER, -EIO, "STATUS_INVALID_QUOTA_LOWER"},
	{STATUS_DISK_CORRUPT_ERROR, -EIO, "STATUS_DISK_CORRUPT_ERROR"},
	{STATUS_OBJECT_NAME_INVALID, -ENOENT, "STATUS_OBJECT_NAME_INVALID"},
	{STATUS_OBJECT_NAME_NOT_FOUND, -ENOENT, "STATUS_OBJECT_NAME_NOT_FOUND"},
	{STATUS_OBJECT_NAME_COLLISION, -EEXIST, "STATUS_OBJECT_NAME_COLLISION"},
	{STATUS_PORT_DISCONNECTED, -EIO, "STATUS_PORT_DISCONNECTED"},
	{STATUS_DEVICE_ALREADY_ATTACHED, -EIO,
	"STATUS_DEVICE_ALREADY_ATTACHED"},
	{STATUS_OBJECT_PATH_INVALID, -ENOTDIR, "STATUS_OBJECT_PATH_INVALID"},
	{STATUS_OBJECT_PATH_NOT_FOUND, -ENOENT, "STATUS_OBJECT_PATH_NOT_FOUND"},
	{STATUS_OBJECT_PATH_SYNTAX_BAD, -EIO, "STATUS_OBJECT_PATH_SYNTAX_BAD"},
	{STATUS_DATA_OVERRUN, -EIO, "STATUS_DATA_OVERRUN"},
	{STATUS_DATA_LATE_ERROR, -EIO, "STATUS_DATA_LATE_ERROR"},
	{STATUS_DATA_ERROR, -EIO, "STATUS_DATA_ERROR"},
	{STATUS_CRC_ERROR, -EIO, "STATUS_CRC_ERROR"},
	{STATUS_SECTION_TOO_BIG, -EIO, "STATUS_SECTION_TOO_BIG"},
	{STATUS_PORT_CONNECTION_REFUSED, -ECONNREFUSED,
	"STATUS_PORT_CONNECTION_REFUSED"},
	{STATUS_INVALID_PORT_HANDLE, -EIO, "STATUS_INVALID_PORT_HANDLE"},
	{STATUS_SHARING_VIOLATION, -EBUSY, "STATUS_SHARING_VIOLATION"},
	{STATUS_QUOTA_EXCEEDED, -EDQUOT, "STATUS_QUOTA_EXCEEDED"},
	{STATUS_INVALID_PAGE_PROTECTION, -EIO,
	"STATUS_INVALID_PAGE_PROTECTION"},
	{STATUS_MUTANT_NOT_OWNED, -EIO, "STATUS_MUTANT_NOT_OWNED"},
	{STATUS_SEMAPHORE_LIMIT_EXCEEDED, -EIO,
	"STATUS_SEMAPHORE_LIMIT_EXCEEDED"},
	{STATUS_PORT_ALREADY_SET, -EIO, "STATUS_PORT_ALREADY_SET"},
	{STATUS_SECTION_NOT_IMAGE, -EIO, "STATUS_SECTION_NOT_IMAGE"},
	{STATUS_SUSPEND_COUNT_EXCEEDED, -EIO, "STATUS_SUSPEND_COUNT_EXCEEDED"},
	{STATUS_THREAD_IS_TERMINATING, -EIO, "STATUS_THREAD_IS_TERMINATING"},
	{STATUS_BAD_WORKING_SET_LIMIT, -EIO, "STATUS_BAD_WORKING_SET_LIMIT"},
	{STATUS_INCOMPATIBLE_FILE_MAP, -EIO, "STATUS_INCOMPATIBLE_FILE_MAP"},
	{STATUS_SECTION_PROTECTION, -EIO, "STATUS_SECTION_PROTECTION"},
	{STATUS_EAS_NOT_SUPPORTED, -EOPNOTSUPP, "STATUS_EAS_NOT_SUPPORTED"},
	{STATUS_EA_TOO_LARGE, -EIO, "STATUS_EA_TOO_LARGE"},
	{STATUS_NONEXISTENT_EA_ENTRY, -EIO, "STATUS_NONEXISTENT_EA_ENTRY"},
	{STATUS_NO_EAS_ON_FILE, -ENODATA, "STATUS_NO_EAS_ON_FILE"},
	{STATUS_EA_CORRUPT_ERROR, -EIO, "STATUS_EA_CORRUPT_ERROR"},
	{STATUS_FILE_LOCK_CONFLICT, -EACCES, "STATUS_FILE_LOCK_CONFLICT"},
	{STATUS_LOCK_NOT_GRANTED, -EACCES, "STATUS_LOCK_NOT_GRANTED"},
	{STATUS_DELETE_PENDING, -ENOENT, "STATUS_DELETE_PENDING"},
	{STATUS_CTL_FILE_NOT_SUPPORTED, -ENOSYS,
	"STATUS_CTL_FILE_NOT_SUPPORTED"},
	{STATUS_UNKNOWN_REVISION, -EIO, "STATUS_UNKNOWN_REVISION"},
	{STATUS_REVISION_MISMATCH, -EIO, "STATUS_REVISION_MISMATCH"},
	{STATUS_INVALID_OWNER, -EIO, "STATUS_INVALID_OWNER"},
	{STATUS_INVALID_PRIMARY_GROUP, -EIO, "STATUS_INVALID_PRIMARY_GROUP"},
	{STATUS_NO_IMPERSONATION_TOKEN, -EIO, "STATUS_NO_IMPERSONATION_TOKEN"},
	{STATUS_CANT_DISABLE_MANDATORY, -EIO, "STATUS_CANT_DISABLE_MANDATORY"},
	{STATUS_NO_LOGON_SERVERS, -EIO, "STATUS_NO_LOGON_SERVERS"},
	{STATUS_NO_SUCH_LOGON_SESSION, -EIO, "STATUS_NO_SUCH_LOGON_SESSION"},
	{STATUS_NO_SUCH_PRIVILEGE, -EIO, "STATUS_NO_SUCH_PRIVILEGE"},
	{STATUS_PRIVILEGE_NOT_HELD, -EIO, "STATUS_PRIVILEGE_NOT_HELD"},
	{STATUS_INVALID_ACCOUNT_NAME, -EIO, "STATUS_INVALID_ACCOUNT_NAME"},
	{STATUS_USER_EXISTS, -EIO, "STATUS_USER_EXISTS"},
	{STATUS_NO_SUCH_USER, -EIO, "STATUS_NO_SUCH_USER"},
	{STATUS_GROUP_EXISTS, -EIO, "STATUS_GROUP_EXISTS"},
	{STATUS_NO_SUCH_GROUP, -EIO, "STATUS_NO_SUCH_GROUP"},
	{STATUS_MEMBER_IN_GROUP, -EIO, "STATUS_MEMBER_IN_GROUP"},
	{STATUS_MEMBER_NOT_IN_GROUP, -EIO, "STATUS_MEMBER_NOT_IN_GROUP"},
	{STATUS_LAST_ADMIN, -EIO, "STATUS_LAST_ADMIN"},
	{STATUS_WRONG_PASSWORD, -EACCES, "STATUS_WRONG_PASSWORD"},
	{STATUS_ILL_FORMED_PASSWORD, -EINVAL, "STATUS_ILL_FORMED_PASSWORD"},
	{STATUS_PASSWORD_RESTRICTION, -EACCES, "STATUS_PASSWORD_RESTRICTION"},
	{STATUS_LOGON_FAILURE, -EACCES, "STATUS_LOGON_FAILURE"},
	{STATUS_ACCOUNT_RESTRICTION, -EACCES, "STATUS_ACCOUNT_RESTRICTION"},
	{STATUS_INVALID_LOGON_HOURS, -EACCES, "STATUS_INVALID_LOGON_HOURS"},
	{STATUS_INVALID_WORKSTATION, -EACCES, "STATUS_INVALID_WORKSTATION"},
	{STATUS_PASSWORD_EXPIRED, -EKEYEXPIRED, "STATUS_PASSWORD_EXPIRED"},
	{STATUS_ACCOUNT_DISABLED, -EKEYREVOKED, "STATUS_ACCOUNT_DISABLED"},
	{STATUS_NONE_MAPPED, -EIO, "STATUS_NONE_MAPPED"},
	{STATUS_TOO_MANY_LUIDS_REQUESTED, -EIO,
	"STATUS_TOO_MANY_LUIDS_REQUESTED"},
	{STATUS_LUIDS_EXHAUSTED, -EIO, "STATUS_LUIDS_EXHAUSTED"},
	{STATUS_INVALID_SUB_AUTHORITY, -EIO, "STATUS_INVALID_SUB_AUTHORITY"},
	{STATUS_INVALID_ACL, -EIO, "STATUS_INVALID_ACL"},
	{STATUS_INVALID_SID, -EIO, "STATUS_INVALID_SID"},
	{STATUS_INVALID_SECURITY_DESCR, -EIO, "STATUS_INVALID_SECURITY_DESCR"},
	{STATUS_PROCEDURE_NOT_FOUND, -EIO, "STATUS_PROCEDURE_NOT_FOUND"},
	{STATUS_INVALID_IMAGE_FORMAT, -EIO, "STATUS_INVALID_IMAGE_FORMAT"},
	{STATUS_NO_TOKEN, -EIO, "STATUS_NO_TOKEN"},
	{STATUS_BAD_INHERITANCE_ACL, -EIO, "STATUS_BAD_INHERITANCE_ACL"},
	{STATUS_RANGE_NOT_LOCKED, -EIO, "STATUS_RANGE_NOT_LOCKED"},
	{STATUS_DISK_FULL, -ENOSPC, "STATUS_DISK_FULL"},
	{STATUS_SERVER_DISABLED, -EIO, "STATUS_SERVER_DISABLED"},
	{STATUS_SERVER_NOT_DISABLED, -EIO, "STATUS_SERVER_NOT_DISABLED"},
	{STATUS_TOO_MANY_GUIDS_REQUESTED, -EIO,
	"STATUS_TOO_MANY_GUIDS_REQUESTED"},
	{STATUS_GUIDS_EXHAUSTED, -EIO, "STATUS_GUIDS_EXHAUSTED"},
	{STATUS_INVALID_ID_AUTHORITY, -EIO, "STATUS_INVALID_ID_AUTHORITY"},
	{STATUS_AGENTS_EXHAUSTED, -EIO, "STATUS_AGENTS_EXHAUSTED"},
	{STATUS_INVALID_VOLUME_LABEL, -EIO, "STATUS_INVALID_VOLUME_LABEL"},
	{STATUS_SECTION_NOT_EXTENDED, -EIO, "STATUS_SECTION_NOT_EXTENDED"},
	{STATUS_NOT_MAPPED_DATA, -EIO, "STATUS_NOT_MAPPED_DATA"},
	{STATUS_RESOURCE_DATA_NOT_FOUND, -EIO,
	"STATUS_RESOURCE_DATA_NOT_FOUND"},
	{STATUS_RESOURCE_TYPE_NOT_FOUND, -EIO,
	"STATUS_RESOURCE_TYPE_NOT_FOUND"},
	{STATUS_RESOURCE_NAME_NOT_FOUND, -EIO,
	"STATUS_RESOURCE_NAME_NOT_FOUND"},
	{STATUS_ARRAY_BOUNDS_EXCEEDED, -EIO, "STATUS_ARRAY_BOUNDS_EXCEEDED"},
	{STATUS_FLOAT_DENORMAL_OPERAND, -EIO, "STATUS_FLOAT_DENORMAL_OPERAND"},
	{STATUS_FLOAT_DIVIDE_BY_ZERO, -EIO, "STATUS_FLOAT_DIVIDE_BY_ZERO"},
	{STATUS_FLOAT_INEXACT_RESULT, -EIO, "STATUS_FLOAT_INEXACT_RESULT"},
	{STATUS_FLOAT_INVALID_OPERATION, -EIO,
	"STATUS_FLOAT_INVALID_OPERATION"},
	{STATUS_FLOAT_OVERFLOW, -EIO, "STATUS_FLOAT_OVERFLOW"},
	{STATUS_FLOAT_STACK_CHECK, -EIO, "STATUS_FLOAT_STACK_CHECK"},
	{STATUS_FLOAT_UNDERFLOW, -EIO, "STATUS_FLOAT_UNDERFLOW"},
	{STATUS_INTEGER_DIVIDE_BY_ZERO, -EIO, "STATUS_INTEGER_DIVIDE_BY_ZERO"},
	{STATUS_INTEGER_OVERFLOW, -EIO, "STATUS_INTEGER_OVERFLOW"},
	{STATUS_PRIVILEGED_INSTRUCTION, -EIO, "STATUS_PRIVILEGED_INSTRUCTION"},
	{STATUS_TOO_MANY_PAGING_FILES, -EIO, "STATUS_TOO_MANY_PAGING_FILES"},
	{STATUS_FILE_INVALID, -EIO, "STATUS_FILE_INVALID"},
	{STATUS_ALLOTTED_SPACE_EXCEEDED, -EIO,
	"STATUS_ALLOTTED_SPACE_EXCEEDED"},
	{STATUS_INSUFFICIENT_RESOURCES, -EREMOTEIO,
				"STATUS_INSUFFICIENT_RESOURCES"},
	{STATUS_DFS_EXIT_PATH_FOUND, -EIO, "STATUS_DFS_EXIT_PATH_FOUND"},
	{STATUS_DEVICE_DATA_ERROR, -EIO, "STATUS_DEVICE_DATA_ERROR"},
	{STATUS_DEVICE_NOT_CONNECTED, -EIO, "STATUS_DEVICE_NOT_CONNECTED"},
	{STATUS_DEVICE_POWER_FAILURE, -EIO, "STATUS_DEVICE_POWER_FAILURE"},
	{STATUS_FREE_VM_NOT_AT_BASE, -EIO, "STATUS_FREE_VM_NOT_AT_BASE"},
	{STATUS_MEMORY_NOT_ALLOCATED, -EFAULT, "STATUS_MEMORY_NOT_ALLOCATED"},
	{STATUS_WORKING_SET_QUOTA, -EIO, "STATUS_WORKING_SET_QUOTA"},
	{STATUS_MEDIA_WRITE_PROTECTED, -EROFS, "STATUS_MEDIA_WRITE_PROTECTED"},
	{STATUS_DEVICE_NOT_READY, -EIO, "STATUS_DEVICE_NOT_READY"},
	{STATUS_INVALID_GROUP_ATTRIBUTES, -EIO,
	"STATUS_INVALID_GROUP_ATTRIBUTES"},
	{STATUS_BAD_IMPERSONATION_LEVEL, -EIO,
	"STATUS_BAD_IMPERSONATION_LEVEL"},
	{STATUS_CANT_OPEN_ANONYMOUS, -EIO, "STATUS_CANT_OPEN_ANONYMOUS"},
	{STATUS_BAD_VALIDATION_CLASS, -EIO, "STATUS_BAD_VALIDATION_CLASS"},
	{STATUS_BAD_TOKEN_TYPE, -EIO, "STATUS_BAD_TOKEN_TYPE"},
	{STATUS_BAD_MASTER_BOOT_RECORD, -EIO, "STATUS_BAD_MASTER_BOOT_RECORD"},
	{STATUS_INSTRUCTION_MISALIGNMENT, -EIO,
	"STATUS_INSTRUCTION_MISALIGNMENT"},
	{STATUS_INSTANCE_NOT_AVAILABLE, -EIO, "STATUS_INSTANCE_NOT_AVAILABLE"},
	{STATUS_PIPE_NOT_AVAILABLE, -EIO, "STATUS_PIPE_NOT_AVAILABLE"},
	{STATUS_INVALID_PIPE_STATE, -EIO, "STATUS_INVALID_PIPE_STATE"},
	{STATUS_PIPE_BUSY, -EBUSY, "STATUS_PIPE_BUSY"},
	{STATUS_ILLEGAL_FUNCTION, -EIO, "STATUS_ILLEGAL_FUNCTION"},
	{STATUS_PIPE_DISCONNECTED, -EPIPE, "STATUS_PIPE_DISCONNECTED"},
	{STATUS_PIPE_CLOSING, -EIO, "STATUS_PIPE_CLOSING"},
	{STATUS_PIPE_CONNECTED, -EIO, "STATUS_PIPE_CONNECTED"},
	{STATUS_PIPE_LISTENING, -EIO, "STATUS_PIPE_LISTENING"},
	{STATUS_INVALID_READ_MODE, -EIO, "STATUS_INVALID_READ_MODE"},
	{STATUS_IO_TIMEOUT, -ETIMEDOUT, "STATUS_IO_TIMEOUT"},
	{STATUS_FILE_FORCED_CLOSED, -EIO, "STATUS_FILE_FORCED_CLOSED"},
	{STATUS_PROFILING_NOT_STARTED, -EIO, "STATUS_PROFILING_NOT_STARTED"},
	{STATUS_PROFILING_NOT_STOPPED, -EIO, "STATUS_PROFILING_NOT_STOPPED"},
	{STATUS_COULD_NOT_INTERPRET, -EIO, "STATUS_COULD_NOT_INTERPRET"},
	{STATUS_FILE_IS_A_DIRECTORY, -EISDIR, "STATUS_FILE_IS_A_DIRECTORY"},
	{STATUS_NOT_SUPPORTED, -EOPNOTSUPP, "STATUS_NOT_SUPPORTED"},
	{STATUS_REMOTE_NOT_LISTENING, -EHOSTDOWN,
	"STATUS_REMOTE_NOT_LISTENING"},
	{STATUS_DUPLICATE_NAME, -ENOTUNIQ, "STATUS_DUPLICATE_NAME"},
	{STATUS_BAD_NETWORK_PATH, -EINVAL, "STATUS_BAD_NETWORK_PATH"},
	{STATUS_NETWORK_BUSY, -EBUSY, "STATUS_NETWORK_BUSY"},
	{STATUS_DEVICE_DOES_NOT_EXIST, -ENODEV, "STATUS_DEVICE_DOES_NOT_EXIST"},
	{STATUS_TOO_MANY_COMMANDS, -EIO, "STATUS_TOO_MANY_COMMANDS"},
	{STATUS_ADAPTER_HARDWARE_ERROR, -EIO, "STATUS_ADAPTER_HARDWARE_ERROR"},
	{STATUS_INVALID_NETWORK_RESPONSE, -EIO,
	"STATUS_INVALID_NETWORK_RESPONSE"},
	{STATUS_UNEXPECTED_NETWORK_ERROR, -EIO,
	"STATUS_UNEXPECTED_NETWORK_ERROR"},
	{STATUS_BAD_REMOTE_ADAPTER, -EIO, "STATUS_BAD_REMOTE_ADAPTER"},
	{STATUS_PRINT_QUEUE_FULL, -EIO, "STATUS_PRINT_QUEUE_FULL"},
	{STATUS_NO_SPOOL_SPACE, -EIO, "STATUS_NO_SPOOL_SPACE"},
	{STATUS_PRINT_CANCELLED, -EIO, "STATUS_PRINT_CANCELLED"},
	{STATUS_NETWORK_NAME_DELETED, -EIO, "STATUS_NETWORK_NAME_DELETED"},
	{STATUS_NETWORK_ACCESS_DENIED, -EACCES, "STATUS_NETWORK_ACCESS_DENIED"},
	{STATUS_BAD_DEVICE_TYPE, -EIO, "STATUS_BAD_DEVICE_TYPE"},
	{STATUS_BAD_NETWORK_NAME, -ENOENT, "STATUS_BAD_NETWORK_NAME"},
	{STATUS_TOO_MANY_NAMES, -EIO, "STATUS_TOO_MANY_NAMES"},
	{STATUS_TOO_MANY_SESSIONS, -EIO, "STATUS_TOO_MANY_SESSIONS"},
	{STATUS_SHARING_PAUSED, -EIO, "STATUS_SHARING_PAUSED"},
	{STATUS_REQUEST_NOT_ACCEPTED, -EIO, "STATUS_REQUEST_NOT_ACCEPTED"},
	{STATUS_REDIRECTOR_PAUSED, -EIO, "STATUS_REDIRECTOR_PAUSED"},
	{STATUS_NET_WRITE_FAULT, -EIO, "STATUS_NET_WRITE_FAULT"},
	{STATUS_PROFILING_AT_LIMIT, -EIO, "STATUS_PROFILING_AT_LIMIT"},
	{STATUS_NOT_SAME_DEVICE, -EXDEV, "STATUS_NOT_SAME_DEVICE"},
	{STATUS_FILE_RENAMED, -EIO, "STATUS_FILE_RENAMED"},
	{STATUS_VIRTUAL_CIRCUIT_CLOSED, -EIO, "STATUS_VIRTUAL_CIRCUIT_CLOSED"},
	{STATUS_NO_SECURITY_ON_OBJECT, -EIO, "STATUS_NO_SECURITY_ON_OBJECT"},
	{STATUS_CANT_WAIT, -EIO, "STATUS_CANT_WAIT"},
	{STATUS_PIPE_EMPTY, -EIO, "STATUS_PIPE_EMPTY"},
	{STATUS_CANT_ACCESS_DOMAIN_INFO, -EIO,
	"STATUS_CANT_ACCESS_DOMAIN_INFO"},
	{STATUS_CANT_TERMINATE_SELF, -EIO, "STATUS_CANT_TERMINATE_SELF"},
	{STATUS_INVALID_SERVER_STATE, -EIO, "STATUS_INVALID_SERVER_STATE"},
	{STATUS_INVALID_DOMAIN_STATE, -EIO, "STATUS_INVALID_DOMAIN_STATE"},
	{STATUS_INVALID_DOMAIN_ROLE, -EIO, "STATUS_INVALID_DOMAIN_ROLE"},
	{STATUS_NO_SUCH_DOMAIN, -EIO, "STATUS_NO_SUCH_DOMAIN"},
	{STATUS_DOMAIN_EXISTS, -EIO, "STATUS_DOMAIN_EXISTS"},
	{STATUS_DOMAIN_LIMIT_EXCEEDED, -EIO, "STATUS_DOMAIN_LIMIT_EXCEEDED"},
	{STATUS_OPLOCK_NOT_GRANTED, -EIO, "STATUS_OPLOCK_NOT_GRANTED"},
	{STATUS_INVALID_OPLOCK_PROTOCOL, -EIO,
	"STATUS_INVALID_OPLOCK_PROTOCOL"},
	{STATUS_INTERNAL_DB_CORRUPTION, -EIO, "STATUS_INTERNAL_DB_CORRUPTION"},
	{STATUS_INTERNAL_ERROR, -EIO, "STATUS_INTERNAL_ERROR"},
	{STATUS_GENERIC_NOT_MAPPED, -EIO, "STATUS_GENERIC_NOT_MAPPED"},
	{STATUS_BAD_DESCRIPTOR_FORMAT, -EIO, "STATUS_BAD_DESCRIPTOR_FORMAT"},
	{STATUS_INVALID_USER_BUFFER, -EIO, "STATUS_INVALID_USER_BUFFER"},
	{STATUS_UNEXPECTED_IO_ERROR, -EIO, "STATUS_UNEXPECTED_IO_ERROR"},
	{STATUS_UNEXPECTED_MM_CREATE_ERR, -EIO,
	"STATUS_UNEXPECTED_MM_CREATE_ERR"},
	{STATUS_UNEXPECTED_MM_MAP_ERROR, -EIO,
	"STATUS_UNEXPECTED_MM_MAP_ERROR"},
	{STATUS_UNEXPECTED_MM_EXTEND_ERR, -EIO,
	"STATUS_UNEXPECTED_MM_EXTEND_ERR"},
	{STATUS_NOT_LOGON_PROCESS, -EIO, "STATUS_NOT_LOGON_PROCESS"},
	{STATUS_LOGON_SESSION_EXISTS, -EIO, "STATUS_LOGON_SESSION_EXISTS"},
	{STATUS_INVALID_PARAMETER_1, -EINVAL, "STATUS_INVALID_PARAMETER_1"},
	{STATUS_INVALID_PARAMETER_2, -EINVAL, "STATUS_INVALID_PARAMETER_2"},
	{STATUS_INVALID_PARAMETER_3, -EINVAL, "STATUS_INVALID_PARAMETER_3"},
	{STATUS_INVALID_PARAMETER_4, -EINVAL, "STATUS_INVALID_PARAMETER_4"},
	{STATUS_INVALID_PARAMETER_5, -EINVAL, "STATUS_INVALID_PARAMETER_5"},
	{STATUS_INVALID_PARAMETER_6, -EINVAL, "STATUS_INVALID_PARAMETER_6"},
	{STATUS_INVALID_PARAMETER_7, -EINVAL, "STATUS_INVALID_PARAMETER_7"},
	{STATUS_INVALID_PARAMETER_8, -EINVAL, "STATUS_INVALID_PARAMETER_8"},
	{STATUS_INVALID_PARAMETER_9, -EINVAL, "STATUS_INVALID_PARAMETER_9"},
	{STATUS_INVALID_PARAMETER_10, -EINVAL, "STATUS_INVALID_PARAMETER_10"},
	{STATUS_INVALID_PARAMETER_11, -EINVAL, "STATUS_INVALID_PARAMETER_11"},
	{STATUS_INVALID_PARAMETER_12, -EINVAL, "STATUS_INVALID_PARAMETER_12"},
	{STATUS_REDIRECTOR_NOT_STARTED, -EIO, "STATUS_REDIRECTOR_NOT_STARTED"},
	{STATUS_REDIRECTOR_STARTED, -EIO, "STATUS_REDIRECTOR_STARTED"},
	{STATUS_STACK_OVERFLOW, -EIO, "STATUS_STACK_OVERFLOW"},
	{STATUS_NO_SUCH_PACKAGE, -EIO, "STATUS_NO_SUCH_PACKAGE"},
	{STATUS_BAD_FUNCTION_TABLE, -EIO, "STATUS_BAD_FUNCTION_TABLE"},
	{STATUS_VARIABLE_NOT_FOUND, -EIO, "STATUS_VARIABLE_NOT_FOUND"},
	{STATUS_DIRECTORY_NOT_EMPTY, -ENOTEMPTY, "STATUS_DIRECTORY_NOT_EMPTY"},
	{STATUS_FILE_CORRUPT_ERROR, -EIO, "STATUS_FILE_CORRUPT_ERROR"},
	{STATUS_NOT_A_DIRECTORY, -ENOTDIR, "STATUS_NOT_A_DIRECTORY"},
	{STATUS_BAD_LOGON_SESSION_STATE, -EIO,
	"STATUS_BAD_LOGON_SESSION_STATE"},
	{STATUS_LOGON_SESSION_COLLISION, -EIO,
	"STATUS_LOGON_SESSION_COLLISION"},
	{STATUS_NAME_TOO_LONG, -ENAMETOOLONG, "STATUS_NAME_TOO_LONG"},
	{STATUS_FILES_OPEN, -EIO, "STATUS_FILES_OPEN"},
	{STATUS_CONNECTION_IN_USE, -EIO, "STATUS_CONNECTION_IN_USE"},
	{STATUS_MESSAGE_NOT_FOUND, -EIO, "STATUS_MESSAGE_NOT_FOUND"},
	{STATUS_PROCESS_IS_TERMINATING, -EIO, "STATUS_PROCESS_IS_TERMINATING"},
	{STATUS_INVALID_LOGON_TYPE, -EIO, "STATUS_INVALID_LOGON_TYPE"},
	{STATUS_NO_GUID_TRANSLATION, -EIO, "STATUS_NO_GUID_TRANSLATION"},
	{STATUS_CANNOT_IMPERSONATE, -EIO, "STATUS_CANNOT_IMPERSONATE"},
	{STATUS_IMAGE_ALREADY_LOADED, -EIO, "STATUS_IMAGE_ALREADY_LOADED"},
	{STATUS_ABIOS_NOT_PRESENT, -EIO, "STATUS_ABIOS_NOT_PRESENT"},
	{STATUS_ABIOS_LID_NOT_EXIST, -EIO, "STATUS_ABIOS_LID_NOT_EXIST"},
	{STATUS_ABIOS_LID_ALREADY_OWNED, -EIO,
	"STATUS_ABIOS_LID_ALREADY_OWNED"},
	{STATUS_ABIOS_NOT_LID_OWNER, -EIO, "STATUS_ABIOS_NOT_LID_OWNER"},
	{STATUS_ABIOS_INVALID_COMMAND, -EIO, "STATUS_ABIOS_INVALID_COMMAND"},
	{STATUS_ABIOS_INVALID_LID, -EIO, "STATUS_ABIOS_INVALID_LID"},
	{STATUS_ABIOS_SELECTOR_NOT_AVAILABLE, -EIO,
	"STATUS_ABIOS_SELECTOR_NOT_AVAILABLE"},
	{STATUS_ABIOS_INVALID_SELECTOR, -EIO, "STATUS_ABIOS_INVALID_SELECTOR"},
	{STATUS_NO_LDT, -EIO, "STATUS_NO_LDT"},
	{STATUS_INVALID_LDT_SIZE, -EIO, "STATUS_INVALID_LDT_SIZE"},
	{STATUS_INVALID_LDT_OFFSET, -EIO, "STATUS_INVALID_LDT_OFFSET"},
	{STATUS_INVALID_LDT_DESCRIPTOR, -EIO, "STATUS_INVALID_LDT_DESCRIPTOR"},
	{STATUS_INVALID_IMAGE_NE_FORMAT, -EIO,
	"STATUS_INVALID_IMAGE_NE_FORMAT"},
	{STATUS_RXACT_INVALID_STATE, -EIO, "STATUS_RXACT_INVALID_STATE"},
	{STATUS_RXACT_COMMIT_FAILURE, -EIO, "STATUS_RXACT_COMMIT_FAILURE"},
	{STATUS_MAPPED_FILE_SIZE_ZERO, -EIO, "STATUS_MAPPED_FILE_SIZE_ZERO"},
	{STATUS_TOO_MANY_OPENED_FILES, -EMFILE, "STATUS_TOO_MANY_OPENED_FILES"},
	{STATUS_CANCELLED, -EIO, "STATUS_CANCELLED"},
	{STATUS_CANNOT_DELETE, -EACCES, "STATUS_CANNOT_DELETE"},
	{STATUS_INVALID_COMPUTER_NAME, -EIO, "STATUS_INVALID_COMPUTER_NAME"},
	{STATUS_FILE_DELETED, -EIO, "STATUS_FILE_DELETED"},
	{STATUS_SPECIAL_ACCOUNT, -EIO, "STATUS_SPECIAL_ACCOUNT"},
	{STATUS_SPECIAL_GROUP, -EIO, "STATUS_SPECIAL_GROUP"},
	{STATUS_SPECIAL_USER, -EIO, "STATUS_SPECIAL_USER"},
	{STATUS_MEMBERS_PRIMARY_GROUP, -EIO, "STATUS_MEMBERS_PRIMARY_GROUP"},
	{STATUS_FILE_CLOSED, -EBADF, "STATUS_FILE_CLOSED"},
	{STATUS_TOO_MANY_THREADS, -EIO, "STATUS_TOO_MANY_THREADS"},
	{STATUS_THREAD_NOT_IN_PROCESS, -EIO, "STATUS_THREAD_NOT_IN_PROCESS"},
	{STATUS_TOKEN_ALREADY_IN_USE, -EIO, "STATUS_TOKEN_ALREADY_IN_USE"},
	{STATUS_PAGEFILE_QUOTA_EXCEEDED, -EDQUOT,
	"STATUS_PAGEFILE_QUOTA_EXCEEDED"},
	{STATUS_COMMITMENT_LIMIT, -EIO, "STATUS_COMMITMENT_LIMIT"},
	{STATUS_INVALID_IMAGE_LE_FORMAT, -EIO,
	"STATUS_INVALID_IMAGE_LE_FORMAT"},
	{STATUS_INVALID_IMAGE_NOT_MZ, -EIO, "STATUS_INVALID_IMAGE_NOT_MZ"},
	{STATUS_INVALID_IMAGE_PROTECT, -EIO, "STATUS_INVALID_IMAGE_PROTECT"},
	{STATUS_INVALID_IMAGE_WIN_16, -EIO, "STATUS_INVALID_IMAGE_WIN_16"},
	{STATUS_LOGON_SERVER_CONFLICT, -EIO, "STATUS_LOGON_SERVER_CONFLICT"},
	{STATUS_TIME_DIFFERENCE_AT_DC, -EIO, "STATUS_TIME_DIFFERENCE_AT_DC"},
	{STATUS_SYNCHRONIZATION_REQUIRED, -EIO,
	"STATUS_SYNCHRONIZATION_REQUIRED"},
	{STATUS_DLL_NOT_FOUND, -ENOENT, "STATUS_DLL_NOT_FOUND"},
	{STATUS_OPEN_FAILED, -EIO, "STATUS_OPEN_FAILED"},
	{STATUS_IO_PRIVILEGE_FAILED, -EIO, "STATUS_IO_PRIVILEGE_FAILED"},
	{STATUS_ORDINAL_NOT_FOUND, -EIO, "STATUS_ORDINAL_NOT_FOUND"},
	{STATUS_ENTRYPOINT_NOT_FOUND, -EIO, "STATUS_ENTRYPOINT_NOT_FOUND"},
	{STATUS_CONTROL_C_EXIT, -EIO, "STATUS_CONTROL_C_EXIT"},
	{STATUS_LOCAL_DISCONNECT, -EIO, "STATUS_LOCAL_DISCONNECT"},
	{STATUS_REMOTE_DISCONNECT, -ESHUTDOWN, "STATUS_REMOTE_DISCONNECT"},
	{STATUS_REMOTE_RESOURCES, -EIO, "STATUS_REMOTE_RESOURCES"},
	{STATUS_LINK_FAILED, -EXDEV, "STATUS_LINK_FAILED"},
	{STATUS_LINK_TIMEOUT, -ETIMEDOUT, "STATUS_LINK_TIMEOUT"},
	{STATUS_INVALID_CONNECTION, -EIO, "STATUS_INVALID_CONNECTION"},
	{STATUS_INVALID_ADDRESS, -EIO, "STATUS_INVALID_ADDRESS"},
	{STATUS_DLL_INIT_FAILED, -EIO, "STATUS_DLL_INIT_FAILED"},
	{STATUS_MISSING_SYSTEMFILE, -EIO, "STATUS_MISSING_SYSTEMFILE"},
	{STATUS_UNHANDLED_EXCEPTION, -EIO, "STATUS_UNHANDLED_EXCEPTION"},
	{STATUS_APP_INIT_FAILURE, -EIO, "STATUS_APP_INIT_FAILURE"},
	{STATUS_PAGEFILE_CREATE_FAILED, -EIO, "STATUS_PAGEFILE_CREATE_FAILED"},
	{STATUS_NO_PAGEFILE, -EIO, "STATUS_NO_PAGEFILE"},
	{STATUS_INVALID_LEVEL, -EIO, "STATUS_INVALID_LEVEL"},
	{STATUS_WRONG_PASSWORD_CORE, -EIO, "STATUS_WRONG_PASSWORD_CORE"},
	{STATUS_ILLEGAL_FLOAT_CONTEXT, -EIO, "STATUS_ILLEGAL_FLOAT_CONTEXT"},
	{STATUS_PIPE_BROKEN, -EPIPE, "STATUS_PIPE_BROKEN"},
	{STATUS_REGISTRY_CORRUPT, -EIO, "STATUS_REGISTRY_CORRUPT"},
	{STATUS_REGISTRY_IO_FAILED, -EIO, "STATUS_REGISTRY_IO_FAILED"},
	{STATUS_NO_EVENT_PAIR, -EIO, "STATUS_NO_EVENT_PAIR"},
	{STATUS_UNRECOGNIZED_VOLUME, -EIO, "STATUS_UNRECOGNIZED_VOLUME"},
	{STATUS_SERIAL_NO_DEVICE_INITED, -EIO,
	"STATUS_SERIAL_NO_DEVICE_INITED"},
	{STATUS_NO_SUCH_ALIAS, -EIO, "STATUS_NO_SUCH_ALIAS"},
	{STATUS_MEMBER_NOT_IN_ALIAS, -EIO, "STATUS_MEMBER_NOT_IN_ALIAS"},
	{STATUS_MEMBER_IN_ALIAS, -EIO, "STATUS_MEMBER_IN_ALIAS"},
	{STATUS_ALIAS_EXISTS, -EIO, "STATUS_ALIAS_EXISTS"},
	{STATUS_LOGON_NOT_GRANTED, -EIO, "STATUS_LOGON_NOT_GRANTED"},
	{STATUS_TOO_MANY_SECRETS, -EIO, "STATUS_TOO_MANY_SECRETS"},
	{STATUS_SECRET_TOO_LONG, -EIO, "STATUS_SECRET_TOO_LONG"},
	{STATUS_INTERNAL_DB_ERROR, -EIO, "STATUS_INTERNAL_DB_ERROR"},
	{STATUS_FULLSCREEN_MODE, -EIO, "STATUS_FULLSCREEN_MODE"},
	{STATUS_TOO_MANY_CONTEXT_IDS, -EIO, "STATUS_TOO_MANY_CONTEXT_IDS"},
	{STATUS_LOGON_TYPE_NOT_GRANTED, -EIO, "STATUS_LOGON_TYPE_NOT_GRANTED"},
	{STATUS_NOT_REGISTRY_FILE, -EIO, "STATUS_NOT_REGISTRY_FILE"},
	{STATUS_NT_CROSS_ENCRYPTION_REQUIRED, -EIO,
	"STATUS_NT_CROSS_ENCRYPTION_REQUIRED"},
	{STATUS_DOMAIN_CTRLR_CONFIG_ERROR, -EIO,
	"STATUS_DOMAIN_CTRLR_CONFIG_ERROR"},
	{STATUS_FT_MISSING_MEMBER, -EIO, "STATUS_FT_MISSING_MEMBER"},
	{STATUS_ILL_FORMED_SERVICE_ENTRY, -EIO,
	"STATUS_ILL_FORMED_SERVICE_ENTRY"},
	{STATUS_ILLEGAL_CHARACTER, -EIO, "STATUS_ILLEGAL_CHARACTER"},
	{STATUS_UNMAPPABLE_CHARACTER, -EIO, "STATUS_UNMAPPABLE_CHARACTER"},
	{STATUS_UNDEFINED_CHARACTER, -EIO, "STATUS_UNDEFINED_CHARACTER"},
	{STATUS_FLOPPY_VOLUME, -EIO, "STATUS_FLOPPY_VOLUME"},
	{STATUS_FLOPPY_ID_MARK_NOT_FOUND, -EIO,
	"STATUS_FLOPPY_ID_MARK_NOT_FOUND"},
	{STATUS_FLOPPY_WRONG_CYLINDER, -EIO, "STATUS_FLOPPY_WRONG_CYLINDER"},
	{STATUS_FLOPPY_UNKNOWN_ERROR, -EIO, "STATUS_FLOPPY_UNKNOWN_ERROR"},
	{STATUS_FLOPPY_BAD_REGISTERS, -EIO, "STATUS_FLOPPY_BAD_REGISTERS"},
	{STATUS_DISK_RECALIBRATE_FAILED, -EIO,
	"STATUS_DISK_RECALIBRATE_FAILED"},
	{STATUS_DISK_OPERATION_FAILED, -EIO, "STATUS_DISK_OPERATION_FAILED"},
	{STATUS_DISK_RESET_FAILED, -EIO, "STATUS_DISK_RESET_FAILED"},
	{STATUS_SHARED_IRQ_BUSY, -EBUSY, "STATUS_SHARED_IRQ_BUSY"},
	{STATUS_FT_ORPHANING, -EIO, "STATUS_FT_ORPHANING"},
	{STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT, -EIO,
	"STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT"},
	{STATUS_PARTITION_FAILURE, -EIO, "STATUS_PARTITION_FAILURE"},
	{STATUS_INVALID_BLOCK_LENGTH, -EIO, "STATUS_INVALID_BLOCK_LENGTH"},
	{STATUS_DEVICE_NOT_PARTITIONED, -EIO, "STATUS_DEVICE_NOT_PARTITIONED"},
	{STATUS_UNABLE_TO_LOCK_MEDIA, -EIO, "STATUS_UNABLE_TO_LOCK_MEDIA"},
	{STATUS_UNABLE_TO_UNLOAD_MEDIA, -EIO, "STATUS_UNABLE_TO_UNLOAD_MEDIA"},
	{STATUS_EOM_OVERFLOW, -EIO, "STATUS_EOM_OVERFLOW"},
	{STATUS_NO_MEDIA, -EIO, "STATUS_NO_MEDIA"},
	{STATUS_NO_SUCH_MEMBER, -EIO, "STATUS_NO_SUCH_MEMBER"},
	{STATUS_INVALID_MEMBER, -EIO, "STATUS_INVALID_MEMBER"},
	{STATUS_KEY_DELETED, -EIO, "STATUS_KEY_DELETED"},
	{STATUS_NO_LOG_SPACE, -EIO, "STATUS_NO_LOG_SPACE"},
	{STATUS_TOO_MANY_SIDS, -EIO, "STATUS_TOO_MANY_SIDS"},
	{STATUS_LM_CROSS_ENCRYPTION_REQUIRED, -EIO,
	"STATUS_LM_CROSS_ENCRYPTION_REQUIRED"},
	{STATUS_KEY_HAS_CHILDREN, -EIO, "STATUS_KEY_HAS_CHILDREN"},
	{STATUS_CHILD_MUST_BE_VOLATILE, -EIO, "STATUS_CHILD_MUST_BE_VOLATILE"},
	{STATUS_DEVICE_CONFIGURATION_ERROR, -EIO,
	"STATUS_DEVICE_CONFIGURATION_ERROR"},
	{STATUS_DRIVER_INTERNAL_ERROR, -EIO, "STATUS_DRIVER_INTERNAL_ERROR"},
	{STATUS_INVALID_DEVICE_STATE, -EIO, "STATUS_INVALID_DEVICE_STATE"},
	{STATUS_IO_DEVICE_ERROR, -EIO, "STATUS_IO_DEVICE_ERROR"},
	{STATUS_DEVICE_PROTOCOL_ERROR, -EIO, "STATUS_DEVICE_PROTOCOL_ERROR"},
	{STATUS_BACKUP_CONTROLLER, -EIO, "STATUS_BACKUP_CONTROLLER"},
	{STATUS_LOG_FILE_FULL, -EIO, "STATUS_LOG_FILE_FULL"},
	{STATUS_TOO_LATE, -EIO, "STATUS_TOO_LATE"},
	{STATUS_NO_TRUST_LSA_SECRET, -EIO, "STATUS_NO_TRUST_LSA_SECRET"},
	{STATUS_NO_TRUST_SAM_ACCOUNT, -EIO, "STATUS_NO_TRUST_SAM_ACCOUNT"},
	{STATUS_TRUSTED_DOMAIN_FAILURE, -EIO, "STATUS_TRUSTED_DOMAIN_FAILURE"},
	{STATUS_TRUSTED_RELATIONSHIP_FAILURE, -EIO,
	"STATUS_TRUSTED_RELATIONSHIP_FAILURE"},
	{STATUS_EVENTLOG_FILE_CORRUPT, -EIO, "STATUS_EVENTLOG_FILE_CORRUPT"},
	{STATUS_EVENTLOG_CANT_START, -EIO, "STATUS_EVENTLOG_CANT_START"},
	{STATUS_TRUST_FAILURE, -EIO, "STATUS_TRUST_FAILURE"},
	{STATUS_MUTANT_LIMIT_EXCEEDED, -EIO, "STATUS_MUTANT_LIMIT_EXCEEDED"},
	{STATUS_NETLOGON_NOT_STARTED, -EIO, "STATUS_NETLOGON_NOT_STARTED"},
	{STATUS_ACCOUNT_EXPIRED, -EKEYEXPIRED, "STATUS_ACCOUNT_EXPIRED"},
	{STATUS_POSSIBLE_DEADLOCK, -EIO, "STATUS_POSSIBLE_DEADLOCK"},
	{STATUS_NETWORK_CREDENTIAL_CONFLICT, -EIO,
	"STATUS_NETWORK_CREDENTIAL_CONFLICT"},
	{STATUS_REMOTE_SESSION_LIMIT, -EIO, "STATUS_REMOTE_SESSION_LIMIT"},
	{STATUS_EVENTLOG_FILE_CHANGED, -EIO, "STATUS_EVENTLOG_FILE_CHANGED"},
	{STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT, -EIO,
	"STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT"},
	{STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT, -EIO,
	"STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT"},
	{STATUS_NOLOGON_SERVER_TRUST_ACCOUNT, -EIO,
	"STATUS_NOLOGON_SERVER_TRUST_ACCOUNT"},
	{STATUS_DOMAIN_TRUST_INCONSISTENT, -EIO,
	"STATUS_DOMAIN_TRUST_INCONSISTENT"},
	{STATUS_FS_DRIVER_REQUIRED, -EIO, "STATUS_FS_DRIVER_REQUIRED"},
	{STATUS_IMAGE_ALREADY_LOADED_AS_DLL, -EIO,
	"STATUS_IMAGE_ALREADY_LOADED_AS_DLL"},
	{STATUS_NETWORK_OPEN_RESTRICTION, -EIO,
	"STATUS_NETWORK_OPEN_RESTRICTION"},
	{STATUS_NO_USER_SESSION_KEY, -EIO, "STATUS_NO_USER_SESSION_KEY"},
	{STATUS_USER_SESSION_DELETED, -EIO, "STATUS_USER_SESSION_DELETED"},
	{STATUS_RESOURCE_LANG_NOT_FOUND, -EIO,
	"STATUS_RESOURCE_LANG_NOT_FOUND"},
	{STATUS_INSUFF_SERVER_RESOURCES, -EIO,
	"STATUS_INSUFF_SERVER_RESOURCES"},
	{STATUS_INVALID_BUFFER_SIZE, -EIO, "STATUS_INVALID_BUFFER_SIZE"},
	{STATUS_INVALID_ADDRESS_COMPONENT, -EIO,
	"STATUS_INVALID_ADDRESS_COMPONENT"},
	{STATUS_INVALID_ADDRESS_WILDCARD, -EIO,
	"STATUS_INVALID_ADDRESS_WILDCARD"},
	{STATUS_TOO_MANY_ADDRESSES, -EIO, "STATUS_TOO_MANY_ADDRESSES"},
	{STATUS_ADDRESS_ALREADY_EXISTS, -EADDRINUSE,
	"STATUS_ADDRESS_ALREADY_EXISTS"},
	{STATUS_ADDRESS_CLOSED, -EIO, "STATUS_ADDRESS_CLOSED"},
	{STATUS_CONNECTION_DISCONNECTED, -ECONNABORTED,
	"STATUS_CONNECTION_DISCONNECTED"},
	{STATUS_CONNECTION_RESET, -ENETRESET, "STATUS_CONNECTION_RESET"},
	{STATUS_TOO_MANY_NODES, -EIO, "STATUS_TOO_MANY_NODES"},
	{STATUS_TRANSACTION_ABORTED, -EIO, "STATUS_TRANSACTION_ABORTED"},
	{STATUS_TRANSACTION_TIMED_OUT, -EIO, "STATUS_TRANSACTION_TIMED_OUT"},
	{STATUS_TRANSACTION_NO_RELEASE, -EIO, "STATUS_TRANSACTION_NO_RELEASE"},
	{STATUS_TRANSACTION_NO_MATCH, -EIO, "STATUS_TRANSACTION_NO_MATCH"},
	{STATUS_TRANSACTION_RESPONDED, -EIO, "STATUS_TRANSACTION_RESPONDED"},
	{STATUS_TRANSACTION_INVALID_ID, -EIO, "STATUS_TRANSACTION_INVALID_ID"},
	{STATUS_TRANSACTION_INVALID_TYPE, -EIO,
	"STATUS_TRANSACTION_INVALID_TYPE"},
	{STATUS_NOT_SERVER_SESSION, -EIO, "STATUS_NOT_SERVER_SESSION"},
	{STATUS_NOT_CLIENT_SESSION, -EIO, "STATUS_NOT_CLIENT_SESSION"},
	{STATUS_CANNOT_LOAD_REGISTRY_FILE, -EIO,
	"STATUS_CANNOT_LOAD_REGISTRY_FILE"},
	{STATUS_DEBUG_ATTACH_FAILED, -EIO, "STATUS_DEBUG_ATTACH_FAILED"},
	{STATUS_SYSTEM_PROCESS_TERMINATED, -EIO,
	"STATUS_SYSTEM_PROCESS_TERMINATED"},
	{STATUS_DATA_NOT_ACCEPTED, -EIO, "STATUS_DATA_NOT_ACCEPTED"},
	{STATUS_NO_BROWSER_SERVERS_FOUND, -EIO,
	"STATUS_NO_BROWSER_SERVERS_FOUND"},
	{STATUS_VDM_HARD_ERROR, -EIO, "STATUS_VDM_HARD_ERROR"},
	{STATUS_DRIVER_CANCEL_TIMEOUT, -EIO, "STATUS_DRIVER_CANCEL_TIMEOUT"},
	{STATUS_REPLY_MESSAGE_MISMATCH, -EIO, "STATUS_REPLY_MESSAGE_MISMATCH"},
	{STATUS_MAPPED_ALIGNMENT, -EIO, "STATUS_MAPPED_ALIGNMENT"},
	{STATUS_IMAGE_CHECKSUM_MISMATCH, -EIO,
	"STATUS_IMAGE_CHECKSUM_MISMATCH"},
	{STATUS_LOST_WRITEBEHIND_DATA, -EIO, "STATUS_LOST_WRITEBEHIND_DATA"},
	{STATUS_CLIENT_SERVER_PARAMETERS_INVALID, -EIO,
	"STATUS_CLIENT_SERVER_PARAMETERS_INVALID"},
	{STATUS_PASSWORD_MUST_CHANGE, -EIO, "STATUS_PASSWORD_MUST_CHANGE"},
	{STATUS_NOT_FOUND, -ENOENT, "STATUS_NOT_FOUND"},
	{STATUS_NOT_TINY_STREAM, -EIO, "STATUS_NOT_TINY_STREAM"},
	{STATUS_RECOVERY_FAILURE, -EIO, "STATUS_RECOVERY_FAILURE"},
	{STATUS_STACK_OVERFLOW_READ, -EIO, "STATUS_STACK_OVERFLOW_READ"},
	{STATUS_FAIL_CHECK, -EIO, "STATUS_FAIL_CHECK"},
	{STATUS_DUPLICATE_OBJECTID, -EIO, "STATUS_DUPLICATE_OBJECTID"},
	{STATUS_OBJECTID_EXISTS, -EIO, "STATUS_OBJECTID_EXISTS"},
	{STATUS_CONVERT_TO_LARGE, -EIO, "STATUS_CONVERT_TO_LARGE"},
	{STATUS_RETRY, -EAGAIN, "STATUS_RETRY"},
	{STATUS_FOUND_OUT_OF_SCOPE, -EIO, "STATUS_FOUND_OUT_OF_SCOPE"},
	{STATUS_ALLOCATE_BUCKET, -EIO, "STATUS_ALLOCATE_BUCKET"},
	{STATUS_PROPSET_NOT_FOUND, -EIO, "STATUS_PROPSET_NOT_FOUND"},
	{STATUS_MARSHALL_OVERFLOW, -EIO, "STATUS_MARSHALL_OVERFLOW"},
	{STATUS_INVALID_VARIANT, -EIO, "STATUS_INVALID_VARIANT"},
	{STATUS_DOMAIN_CONTROLLER_NOT_FOUND, -EIO,
	"STATUS_DOMAIN_CONTROLLER_NOT_FOUND"},
	{STATUS_ACCOUNT_LOCKED_OUT, -EIO, "STATUS_ACCOUNT_LOCKED_OUT"},
	{STATUS_HANDLE_NOT_CLOSABLE, -EIO, "STATUS_HANDLE_NOT_CLOSABLE"},
	{STATUS_CONNECTION_REFUSED, -EIO, "STATUS_CONNECTION_REFUSED"},
	{STATUS_GRACEFUL_DISCONNECT, -EIO, "STATUS_GRACEFUL_DISCONNECT"},
	{STATUS_ADDRESS_ALREADY_ASSOCIATED, -EIO,
	"STATUS_ADDRESS_ALREADY_ASSOCIATED"},
	{STATUS_ADDRESS_NOT_ASSOCIATED, -EIO, "STATUS_ADDRESS_NOT_ASSOCIATED"},
	{STATUS_CONNECTION_INVALID, -EIO, "STATUS_CONNECTION_INVALID"},
	{STATUS_CONNECTION_ACTIVE, -EIO, "STATUS_CONNECTION_ACTIVE"},
	{STATUS_NETWORK_UNREACHABLE, -ENETUNREACH,
	"STATUS_NETWORK_UNREACHABLE"},
	{STATUS_HOST_UNREACHABLE, -EHOSTDOWN, "STATUS_HOST_UNREACHABLE"},
	{STATUS_PROTOCOL_UNREACHABLE, -ENETUNREACH,
	"STATUS_PROTOCOL_UNREACHABLE"},
	{STATUS_PORT_UNREACHABLE, -ENETUNREACH, "STATUS_PORT_UNREACHABLE"},
	{STATUS_REQUEST_ABORTED, -EIO, "STATUS_REQUEST_ABORTED"},
	{STATUS_CONNECTION_ABORTED, -ECONNABORTED, "STATUS_CONNECTION_ABORTED"},
	{STATUS_BAD_COMPRESSION_BUFFER, -EIO, "STATUS_BAD_COMPRESSION_BUFFER"},
	{STATUS_USER_MAPPED_FILE, -EIO, "STATUS_USER_MAPPED_FILE"},
	{STATUS_AUDIT_FAILED, -EIO, "STATUS_AUDIT_FAILED"},
	{STATUS_TIMER_RESOLUTION_NOT_SET, -EIO,
	"STATUS_TIMER_RESOLUTION_NOT_SET"},
	{STATUS_CONNECTION_COUNT_LIMIT, -EIO, "STATUS_CONNECTION_COUNT_LIMIT"},
	{STATUS_LOGIN_TIME_RESTRICTION, -EACCES,
	"STATUS_LOGIN_TIME_RESTRICTION"},
	{STATUS_LOGIN_WKSTA_RESTRICTION, -EACCES,
	"STATUS_LOGIN_WKSTA_RESTRICTION"},
	{STATUS_IMAGE_MP_UP_MISMATCH, -EIO, "STATUS_IMAGE_MP_UP_MISMATCH"},
	{STATUS_INSUFFICIENT_LOGON_INFO, -EIO,
	"STATUS_INSUFFICIENT_LOGON_INFO"},
	{STATUS_BAD_DLL_ENTRYPOINT, -EIO, "STATUS_BAD_DLL_ENTRYPOINT"},
	{STATUS_BAD_SERVICE_ENTRYPOINT, -EIO, "STATUS_BAD_SERVICE_ENTRYPOINT"},
	{STATUS_LPC_REPLY_LOST, -EIO, "STATUS_LPC_REPLY_LOST"},
	{STATUS_IP_ADDRESS_CONFLICT1, -EIO, "STATUS_IP_ADDRESS_CONFLICT1"},
	{STATUS_IP_ADDRESS_CONFLICT2, -EIO, "STATUS_IP_ADDRESS_CONFLICT2"},
	{STATUS_REGISTRY_QUOTA_LIMIT, -EDQUOT, "STATUS_REGISTRY_QUOTA_LIMIT"},
	{STATUS_PATH_NOT_COVERED, -EREMOTE, "STATUS_PATH_NOT_COVERED"},
	{STATUS_NO_CALLBACK_ACTIVE, -EIO, "STATUS_NO_CALLBACK_ACTIVE"},
	{STATUS_LICENSE_QUOTA_EXCEEDED, -EACCES,
	"STATUS_LICENSE_QUOTA_EXCEEDED"},
	{STATUS_PWD_TOO_SHORT, -EIO, "STATUS_PWD_TOO_SHORT"},
	{STATUS_PWD_TOO_RECENT, -EIO, "STATUS_PWD_TOO_RECENT"},
	{STATUS_PWD_HISTORY_CONFLICT, -EIO, "STATUS_PWD_HISTORY_CONFLICT"},
	{STATUS_PLUGPLAY_NO_DEVICE, -EIO, "STATUS_PLUGPLAY_NO_DEVICE"},
	{STATUS_UNSUPPORTED_COMPRESSION, -EIO,
	"STATUS_UNSUPPORTED_COMPRESSION"},
	{STATUS_INVALID_HW_PROFILE, -EIO, "STATUS_INVALID_HW_PROFILE"},
	{STATUS_INVALID_PLUGPLAY_DEVICE_PATH, -EIO,
	"STATUS_INVALID_PLUGPLAY_DEVICE_PATH"},
	{STATUS_DRIVER_ORDINAL_NOT_FOUND, -EIO,
	"STATUS_DRIVER_ORDINAL_NOT_FOUND"},
	{STATUS_DRIVER_ENTRYPOINT_NOT_FOUND, -EIO,
	"STATUS_DRIVER_ENTRYPOINT_NOT_FOUND"},
	{STATUS_RESOURCE_NOT_OWNED, -EIO, "STATUS_RESOURCE_NOT_OWNED"},
	{STATUS_TOO_MANY_LINKS, -EMLINK, "STATUS_TOO_MANY_LINKS"},
	{STATUS_QUOTA_LIST_INCONSISTENT, -EIO,
	"STATUS_QUOTA_LIST_INCONSISTENT"},
	{STATUS_FILE_IS_OFFLINE, -EIO, "STATUS_FILE_IS_OFFLINE"},
	{STATUS_EVALUATION_EXPIRATION, -EIO, "STATUS_EVALUATION_EXPIRATION"},
	{STATUS_ILLEGAL_DLL_RELOCATION, -EIO, "STATUS_ILLEGAL_DLL_RELOCATION"},
	{STATUS_LICENSE_VIOLATION, -EIO, "STATUS_LICENSE_VIOLATION"},
	{STATUS_DLL_INIT_FAILED_LOGOFF, -EIO, "STATUS_DLL_INIT_FAILED_LOGOFF"},
	{STATUS_DRIVER_UNABLE_TO_LOAD, -EIO, "STATUS_DRIVER_UNABLE_TO_LOAD"},
	{STATUS_DFS_UNAVAILABLE, -EIO, "STATUS_DFS_UNAVAILABLE"},
	{STATUS_VOLUME_DISMOUNTED, -EIO, "STATUS_VOLUME_DISMOUNTED"},
	{STATUS_WX86_INTERNAL_ERROR, -EIO, "STATUS_WX86_INTERNAL_ERROR"},
	{STATUS_WX86_FLOAT_STACK_CHECK, -EIO, "STATUS_WX86_FLOAT_STACK_CHECK"},
	{STATUS_VALIDATE_CONTINUE, -EIO, "STATUS_VALIDATE_CONTINUE"},
	{STATUS_NO_MATCH, -EIO, "STATUS_NO_MATCH"},
	{STATUS_NO_MORE_MATCHES, -EIO, "STATUS_NO_MORE_MATCHES"},
	{STATUS_NOT_A_REPARSE_POINT, -EIO, "STATUS_NOT_A_REPARSE_POINT"},
	{STATUS_IO_REPARSE_TAG_INVALID, -EIO, "STATUS_IO_REPARSE_TAG_INVALID"},
	{STATUS_IO_REPARSE_TAG_MISMATCH, -EIO,
	"STATUS_IO_REPARSE_TAG_MISMATCH"},
	{STATUS_IO_REPARSE_DATA_INVALID, -EIO,
	"STATUS_IO_REPARSE_DATA_INVALID"},
	{STATUS_IO_REPARSE_TAG_NOT_HANDLED, -EIO,
	"STATUS_IO_REPARSE_TAG_NOT_HANDLED"},
	{STATUS_REPARSE_POINT_NOT_RESOLVED, -EIO,
	"STATUS_REPARSE_POINT_NOT_RESOLVED"},
	{STATUS_DIRECTORY_IS_A_REPARSE_POINT, -EIO,
	"STATUS_DIRECTORY_IS_A_REPARSE_POINT"},
	{STATUS_RANGE_LIST_CONFLICT, -EIO, "STATUS_RANGE_LIST_CONFLICT"},
	{STATUS_SOURCE_ELEMENT_EMPTY, -EIO, "STATUS_SOURCE_ELEMENT_EMPTY"},
	{STATUS_DESTINATION_ELEMENT_FULL, -EIO,
	"STATUS_DESTINATION_ELEMENT_FULL"},
	{STATUS_ILLEGAL_ELEMENT_ADDRESS, -EIO,
	"STATUS_ILLEGAL_ELEMENT_ADDRESS"},
	{STATUS_MAGAZINE_NOT_PRESENT, -EIO, "STATUS_MAGAZINE_NOT_PRESENT"},
	{STATUS_REINITIALIZATION_NEEDED, -EIO,
	"STATUS_REINITIALIZATION_NEEDED"},
	{STATUS_ENCRYPTION_FAILED, -EIO, "STATUS_ENCRYPTION_FAILED"},
	{STATUS_DECRYPTION_FAILED, -EIO, "STATUS_DECRYPTION_FAILED"},
	{STATUS_RANGE_NOT_FOUND, -EIO, "STATUS_RANGE_NOT_FOUND"},
	{STATUS_NO_RECOVERY_POLICY, -EIO, "STATUS_NO_RECOVERY_POLICY"},
	{STATUS_NO_EFS, -EIO, "STATUS_NO_EFS"},
	{STATUS_WRONG_EFS, -EIO, "STATUS_WRONG_EFS"},
	{STATUS_NO_USER_KEYS, -EIO, "STATUS_NO_USER_KEYS"},
	{STATUS_FILE_NOT_ENCRYPTED, -EIO, "STATUS_FILE_NOT_ENCRYPTED"},
	{STATUS_NOT_EXPORT_FORMAT, -EIO, "STATUS_NOT_EXPORT_FORMAT"},
	{STATUS_FILE_ENCRYPTED, -EIO, "STATUS_FILE_ENCRYPTED"},
	{STATUS_WMI_GUID_NOT_FOUND, -EIO, "STATUS_WMI_GUID_NOT_FOUND"},
	{STATUS_WMI_INSTANCE_NOT_FOUND, -EIO, "STATUS_WMI_INSTANCE_NOT_FOUND"},
	{STATUS_WMI_ITEMID_NOT_FOUND, -EIO, "STATUS_WMI_ITEMID_NOT_FOUND"},
	{STATUS_WMI_TRY_AGAIN, -EIO, "STATUS_WMI_TRY_AGAIN"},
	{STATUS_SHARED_POLICY, -EIO, "STATUS_SHARED_POLICY"},
	{STATUS_POLICY_OBJECT_NOT_FOUND, -EIO,
	"STATUS_POLICY_OBJECT_NOT_FOUND"},
	{STATUS_POLICY_ONLY_IN_DS, -EIO, "STATUS_POLICY_ONLY_IN_DS"},
	{STATUS_VOLUME_NOT_UPGRADED, -EIO, "STATUS_VOLUME_NOT_UPGRADED"},
	{STATUS_REMOTE_STORAGE_NOT_ACTIVE, -EIO,
	"STATUS_REMOTE_STORAGE_NOT_ACTIVE"},
	{STATUS_REMOTE_STORAGE_MEDIA_ERROR, -EIO,
	"STATUS_REMOTE_STORAGE_MEDIA_ERROR"},
	{STATUS_NO_TRACKING_SERVICE, -EIO, "STATUS_NO_TRACKING_SERVICE"},
	{STATUS_SERVER_SID_MISMATCH, -EIO, "STATUS_SERVER_SID_MISMATCH"},
	{STATUS_DS_NO_ATTRIBUTE_OR_VALUE, -EIO,
	"STATUS_DS_NO_ATTRIBUTE_OR_VALUE"},
	{STATUS_DS_INVALID_ATTRIBUTE_SYNTAX, -EIO,
	"STATUS_DS_INVALID_ATTRIBUTE_SYNTAX"},
	{STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED, -EIO,
	"STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED"},
	{STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS, -EIO,
	"STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS"},
	{STATUS_DS_BUSY, -EBUSY, "STATUS_DS_BUSY"},
	{STATUS_DS_UNAVAILABLE, -EIO, "STATUS_DS_UNAVAILABLE"},
	{STATUS_DS_NO_RIDS_ALLOCATED, -EIO, "STATUS_DS_NO_RIDS_ALLOCATED"},
	{STATUS_DS_NO_MORE_RIDS, -EIO, "STATUS_DS_NO_MORE_RIDS"},
	{STATUS_DS_INCORRECT_ROLE_OWNER, -EIO,
	"STATUS_DS_INCORRECT_ROLE_OWNER"},
	{STATUS_DS_RIDMGR_INIT_ERROR, -EIO, "STATUS_DS_RIDMGR_INIT_ERROR"},
	{STATUS_DS_OBJ_CLASS_VIOLATION, -EIO, "STATUS_DS_OBJ_CLASS_VIOLATION"},
	{STATUS_DS_CANT_ON_NON_LEAF, -EIO, "STATUS_DS_CANT_ON_NON_LEAF"},
	{STATUS_DS_CANT_ON_RDN, -EIO, "STATUS_DS_CANT_ON_RDN"},
	{STATUS_DS_CANT_MOD_OBJ_CLASS, -EIO, "STATUS_DS_CANT_MOD_OBJ_CLASS"},
	{STATUS_DS_CROSS_DOM_MOVE_FAILED, -EIO,
	"STATUS_DS_CROSS_DOM_MOVE_FAILED"},
	{STATUS_DS_GC_NOT_AVAILABLE, -EIO, "STATUS_DS_GC_NOT_AVAILABLE"},
	{STATUS_DIRECTORY_SERVICE_REQUIRED, -EIO,
	"STATUS_DIRECTORY_SERVICE_REQUIRED"},
	{STATUS_REPARSE_ATTRIBUTE_CONFLICT, -EIO,
	"STATUS_REPARSE_ATTRIBUTE_CONFLICT"},
	{STATUS_CANT_ENABLE_DENY_ONLY, -EIO, "STATUS_CANT_ENABLE_DENY_ONLY"},
	{STATUS_FLOAT_MULTIPLE_FAULTS, -EIO, "STATUS_FLOAT_MULTIPLE_FAULTS"},
	{STATUS_FLOAT_MULTIPLE_TRAPS, -EIO, "STATUS_FLOAT_MULTIPLE_TRAPS"},
	{STATUS_DEVICE_REMOVED, -EIO, "STATUS_DEVICE_REMOVED"},
	{STATUS_JOURNAL_DELETE_IN_PROGRESS, -EIO,
	"STATUS_JOURNAL_DELETE_IN_PROGRESS"},
	{STATUS_JOURNAL_NOT_ACTIVE, -EIO, "STATUS_JOURNAL_NOT_ACTIVE"},
	{STATUS_NOINTERFACE, -EIO, "STATUS_NOINTERFACE"},
	{STATUS_DS_ADMIN_LIMIT_EXCEEDED, -EIO,
	"STATUS_DS_ADMIN_LIMIT_EXCEEDED"},
	{STATUS_DRIVER_FAILED_SLEEP, -EIO, "STATUS_DRIVER_FAILED_SLEEP"},
	{STATUS_MUTUAL_AUTHENTICATION_FAILED, -EIO,
	"STATUS_MUTUAL_AUTHENTICATION_FAILED"},
	{STATUS_CORRUPT_SYSTEM_FILE, -EIO, "STATUS_CORRUPT_SYSTEM_FILE"},
	{STATUS_DATATYPE_MISALIGNMENT_ERROR, -EIO,
	"STATUS_DATATYPE_MISALIGNMENT_ERROR"},
	{STATUS_WMI_READ_ONLY, -EROFS, "STATUS_WMI_READ_ONLY"},
	{STATUS_WMI_SET_FAILURE, -EIO, "STATUS_WMI_SET_FAILURE"},
	{STATUS_COMMITMENT_MINIMUM, -EIO, "STATUS_COMMITMENT_MINIMUM"},
	{STATUS_REG_NAT_CONSUMPTION, -EIO, "STATUS_REG_NAT_CONSUMPTION"},
	{STATUS_TRANSPORT_FULL, -EIO, "STATUS_TRANSPORT_FULL"},
	{STATUS_DS_SAM_INIT_FAILURE, -EIO, "STATUS_DS_SAM_INIT_FAILURE"},
	{STATUS_ONLY_IF_CONNECTED, -EIO, "STATUS_ONLY_IF_CONNECTED"},
	{STATUS_DS_SENSITIVE_GROUP_VIOLATION, -EIO,
	"STATUS_DS_SENSITIVE_GROUP_VIOLATION"},
	{STATUS_PNP_RESTART_ENUMERATION, -EIO,
	"STATUS_PNP_RESTART_ENUMERATION"},
	{STATUS_JOURNAL_ENTRY_DELETED, -EIO, "STATUS_JOURNAL_ENTRY_DELETED"},
	{STATUS_DS_CANT_MOD_PRIMARYGROUPID, -EIO,
	"STATUS_DS_CANT_MOD_PRIMARYGROUPID"},
	{STATUS_SYSTEM_IMAGE_BAD_SIGNATURE, -EIO,
	"STATUS_SYSTEM_IMAGE_BAD_SIGNATURE"},
	{STATUS_PNP_REBOOT_REQUIRED, -EIO, "STATUS_PNP_REBOOT_REQUIRED"},
	{STATUS_POWER_STATE_INVALID, -EIO, "STATUS_POWER_STATE_INVALID"},
	{STATUS_DS_INVALID_GROUP_TYPE, -EIO, "STATUS_DS_INVALID_GROUP_TYPE"},
	{STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN, -EIO,
	"STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN"},
	{STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN, -EIO,
	"STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN"},
	{STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER, -EIO,
	"STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER"},
	{STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER, -EIO,
	"STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER"},
	{STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER, -EIO,
	"STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER"},
	{STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER, -EIO,
	"STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER"},
	{STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER, -EIO,
	"STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER"},
	{STATUS_DS_HAVE_PRIMARY_MEMBERS, -EIO,
	"STATUS_DS_HAVE_PRIMARY_MEMBERS"},
	{STATUS_WMI_NOT_SUPPORTED, -EOPNOTSUPP, "STATUS_WMI_NOT_SUPPORTED"},
	{STATUS_INSUFFICIENT_POWER, -EIO, "STATUS_INSUFFICIENT_POWER"},
	{STATUS_SAM_NEED_BOOTKEY_PASSWORD, -EIO,
	"STATUS_SAM_NEED_BOOTKEY_PASSWORD"},
	{STATUS_SAM_NEED_BOOTKEY_FLOPPY, -EIO,
	"STATUS_SAM_NEED_BOOTKEY_FLOPPY"},
	{STATUS_DS_CANT_START, -EIO, "STATUS_DS_CANT_START"},
	{STATUS_DS_INIT_FAILURE, -EIO, "STATUS_DS_INIT_FAILURE"},
	{STATUS_SAM_INIT_FAILURE, -EIO, "STATUS_SAM_INIT_FAILURE"},
	{STATUS_DS_GC_REQUIRED, -EIO, "STATUS_DS_GC_REQUIRED"},
	{STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY, -EIO,
	"STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY"},
	{STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS, -EIO,
	"STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS"},
	{STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED, -EDQUOT,
	"STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED"},
	{STATUS_MULTIPLE_FAULT_VIOLATION, -EIO,
	"STATUS_MULTIPLE_FAULT_VIOLATION"},
	{STATUS_CURRENT_DOMAIN_NOT_ALLOWED, -EIO,
	"STATUS_CURRENT_DOMAIN_NOT_ALLOWED"},
	{STATUS_CANNOT_MAKE, -EIO, "STATUS_CANNOT_MAKE"},
	{STATUS_SYSTEM_SHUTDOWN, -EIO, "STATUS_SYSTEM_SHUTDOWN"},
	{STATUS_DS_INIT_FAILURE_CONSOLE, -EIO,
	"STATUS_DS_INIT_FAILURE_CONSOLE"},
	{STATUS_DS_SAM_INIT_FAILURE_CONSOLE, -EIO,
	"STATUS_DS_SAM_INIT_FAILURE_CONSOLE"},
	{STATUS_UNFINISHED_CONTEXT_DELETED, -EIO,
	"STATUS_UNFINISHED_CONTEXT_DELETED"},
	{STATUS_NO_TGT_REPLY, -EIO, "STATUS_NO_TGT_REPLY"},
	{STATUS_OBJECTID_NOT_FOUND, -EIO, "STATUS_OBJECTID_NOT_FOUND"},
	{STATUS_NO_IP_ADDRESSES, -EIO, "STATUS_NO_IP_ADDRESSES"},
	{STATUS_WRONG_CREDENTIAL_HANDLE, -EIO,
	"STATUS_WRONG_CREDENTIAL_HANDLE"},
	{STATUS_CRYPTO_SYSTEM_INVALID, -EIO, "STATUS_CRYPTO_SYSTEM_INVALID"},
	{STATUS_MAX_REFERRALS_EXCEEDED, -EIO, "STATUS_MAX_REFERRALS_EXCEEDED"},
	{STATUS_MUST_BE_KDC, -EIO, "STATUS_MUST_BE_KDC"},
	{STATUS_STRONG_CRYPTO_NOT_SUPPORTED, -EIO,
	"STATUS_STRONG_CRYPTO_NOT_SUPPORTED"},
	{STATUS_TOO_MANY_PRINCIPALS, -EIO, "STATUS_TOO_MANY_PRINCIPALS"},
	{STATUS_NO_PA_DATA, -EIO, "STATUS_NO_PA_DATA"},
	{STATUS_PKINIT_NAME_MISMATCH, -EIO, "STATUS_PKINIT_NAME_MISMATCH"},
	{STATUS_SMARTCARD_LOGON_REQUIRED, -EIO,
	"STATUS_SMARTCARD_LOGON_REQUIRED"},
	{STATUS_KDC_INVALID_REQUEST, -EIO, "STATUS_KDC_INVALID_REQUEST"},
	{STATUS_KDC_UNABLE_TO_REFER, -EIO, "STATUS_KDC_UNABLE_TO_REFER"},
	{STATUS_KDC_UNKNOWN_ETYPE, -EIO, "STATUS_KDC_UNKNOWN_ETYPE"},
	{STATUS_SHUTDOWN_IN_PROGRESS, -EIO, "STATUS_SHUTDOWN_IN_PROGRESS"},
	{STATUS_SERVER_SHUTDOWN_IN_PROGRESS, -EIO,
	"STATUS_SERVER_SHUTDOWN_IN_PROGRESS"},
	{STATUS_NOT_SUPPORTED_ON_SBS, -EOPNOTSUPP,
	"STATUS_NOT_SUPPORTED_ON_SBS"},
	{STATUS_WMI_GUID_DISCONNECTED, -EIO, "STATUS_WMI_GUID_DISCONNECTED"},
	{STATUS_WMI_ALREADY_DISABLED, -EIO, "STATUS_WMI_ALREADY_DISABLED"},
	{STATUS_WMI_ALREADY_ENABLED, -EIO, "STATUS_WMI_ALREADY_ENABLED"},
	{STATUS_MFT_TOO_FRAGMENTED, -EIO, "STATUS_MFT_TOO_FRAGMENTED"},
	{STATUS_COPY_PROTECTION_FAILURE, -EIO,
	"STATUS_COPY_PROTECTION_FAILURE"},
	{STATUS_CSS_AUTHENTICATION_FAILURE, -EIO,
	"STATUS_CSS_AUTHENTICATION_FAILURE"},
	{STATUS_CSS_KEY_NOT_PRESENT, -EIO, "STATUS_CSS_KEY_NOT_PRESENT"},
	{STATUS_CSS_KEY_NOT_ESTABLISHED, -EIO,
	"STATUS_CSS_KEY_NOT_ESTABLISHED"},
	{STATUS_CSS_SCRAMBLED_SECTOR, -EIO, "STATUS_CSS_SCRAMBLED_SECTOR"},
	{STATUS_CSS_REGION_MISMATCH, -EIO, "STATUS_CSS_REGION_MISMATCH"},
	{STATUS_CSS_RESETS_EXHAUSTED, -EIO, "STATUS_CSS_RESETS_EXHAUSTED"},
	{STATUS_PKINIT_FAILURE, -EIO, "STATUS_PKINIT_FAILURE"},
	{STATUS_SMARTCARD_SUBSYSTEM_FAILURE, -EIO,
	"STATUS_SMARTCARD_SUBSYSTEM_FAILURE"},
	{STATUS_NO_KERB_KEY, -EIO, "STATUS_NO_KERB_KEY"},
	{STATUS_HOST_DOWN, -EIO, "STATUS_HOST_DOWN"},
	{STATUS_UNSUPPORTED_PREAUTH, -EIO, "STATUS_UNSUPPORTED_PREAUTH"},
	{STATUS_EFS_ALG_BLOB_TOO_BIG, -EIO, "STATUS_EFS_ALG_BLOB_TOO_BIG"},
	{STATUS_PORT_NOT_SET, -EIO, "STATUS_PORT_NOT_SET"},
	{STATUS_DEBUGGER_INACTIVE, -EIO, "STATUS_DEBUGGER_INACTIVE"},
	{STATUS_DS_VERSION_CHECK_FAILURE, -EIO,
	"STATUS_DS_VERSION_CHECK_FAILURE"},
	{STATUS_AUDITING_DISABLED, -EIO, "STATUS_AUDITING_DISABLED"},
	{STATUS_PRENT4_MACHINE_ACCOUNT, -EIO, "STATUS_PRENT4_MACHINE_ACCOUNT"},
	{STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER, -EIO,
	"STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER"},
	{STATUS_INVALID_IMAGE_WIN_32, -EIO, "STATUS_INVALID_IMAGE_WIN_32"},
	{STATUS_INVALID_IMAGE_WIN_64, -EIO, "STATUS_INVALID_IMAGE_WIN_64"},
	{STATUS_BAD_BINDINGS, -EIO, "STATUS_BAD_BINDINGS"},
	{STATUS_NETWORK_SESSION_EXPIRED, -EIO,
	"STATUS_NETWORK_SESSION_EXPIRED"},
	{STATUS_APPHELP_BLOCK, -EIO, "STATUS_APPHELP_BLOCK"},
	{STATUS_ALL_SIDS_FILTERED, -EIO, "STATUS_ALL_SIDS_FILTERED"},
	{STATUS_NOT_SAFE_MODE_DRIVER, -EIO, "STATUS_NOT_SAFE_MODE_DRIVER"},
	{STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT, -EACCES,
	"STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT"},
	{STATUS_ACCESS_DISABLED_BY_POLICY_PATH, -EACCES,
	"STATUS_ACCESS_DISABLED_BY_POLICY_PATH"},
	{STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER, -EACCES,
	"STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER"},
	{STATUS_ACCESS_DISABLED_BY_POLICY_OTHER, -EACCES,
	"STATUS_ACCESS_DISABLED_BY_POLICY_OTHER"},
	{STATUS_FAILED_DRIVER_ENTRY, -EIO, "STATUS_FAILED_DRIVER_ENTRY"},
	{STATUS_DEVICE_ENUMERATION_ERROR, -EIO,
	"STATUS_DEVICE_ENUMERATION_ERROR"},
	{STATUS_MOUNT_POINT_NOT_RESOLVED, -EIO,
	"STATUS_MOUNT_POINT_NOT_RESOLVED"},
	{STATUS_INVALID_DEVICE_OBJECT_PARAMETER, -EIO,
	"STATUS_INVALID_DEVICE_OBJECT_PARAMETER"},
	{STATUS_MCA_OCCURED, -EIO, "STATUS_MCA_OCCURED"},
	{STATUS_DRIVER_BLOCKED_CRITICAL, -EIO,
	"STATUS_DRIVER_BLOCKED_CRITICAL"},
	{STATUS_DRIVER_BLOCKED, -EIO, "STATUS_DRIVER_BLOCKED"},
	{STATUS_DRIVER_DATABASE_ERROR, -EIO, "STATUS_DRIVER_DATABASE_ERROR"},
	{STATUS_SYSTEM_HIVE_TOO_LARGE, -EIO, "STATUS_SYSTEM_HIVE_TOO_LARGE"},
	{STATUS_INVALID_IMPORT_OF_NON_DLL, -EIO,
	"STATUS_INVALID_IMPORT_OF_NON_DLL"},
	{STATUS_NO_SECRETS, -EIO, "STATUS_NO_SECRETS"},
	{STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY, -EACCES,
	"STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY"},
	{STATUS_FAILED_STACK_SWITCH, -EIO, "STATUS_FAILED_STACK_SWITCH"},
	{STATUS_HEAP_CORRUPTION, -EIO, "STATUS_HEAP_CORRUPTION"},
	{STATUS_SMARTCARD_WRONG_PIN, -EIO, "STATUS_SMARTCARD_WRONG_PIN"},
	{STATUS_SMARTCARD_CARD_BLOCKED, -EIO, "STATUS_SMARTCARD_CARD_BLOCKED"},
	{STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED, -EIO,
	"STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED"},
	{STATUS_SMARTCARD_NO_CARD, -EIO, "STATUS_SMARTCARD_NO_CARD"},
	{STATUS_SMARTCARD_NO_KEY_CONTAINER, -EIO,
	"STATUS_SMARTCARD_NO_KEY_CONTAINER"},
	{STATUS_SMARTCARD_NO_CERTIFICATE, -EIO,
	"STATUS_SMARTCARD_NO_CERTIFICATE"},
	{STATUS_SMARTCARD_NO_KEYSET, -EIO, "STATUS_SMARTCARD_NO_KEYSET"},
	{STATUS_SMARTCARD_IO_ERROR, -EIO, "STATUS_SMARTCARD_IO_ERROR"},
	{STATUS_DOWNGRADE_DETECTED, -EIO, "STATUS_DOWNGRADE_DETECTED"},
	{STATUS_SMARTCARD_CERT_REVOKED, -EIO, "STATUS_SMARTCARD_CERT_REVOKED"},
	{STATUS_ISSUING_CA_UNTRUSTED, -EIO, "STATUS_ISSUING_CA_UNTRUSTED"},
	{STATUS_REVOCATION_OFFLINE_C, -EIO, "STATUS_REVOCATION_OFFLINE_C"},
	{STATUS_PKINIT_CLIENT_FAILURE, -EIO, "STATUS_PKINIT_CLIENT_FAILURE"},
	{STATUS_SMARTCARD_CERT_EXPIRED, -EIO, "STATUS_SMARTCARD_CERT_EXPIRED"},
	{STATUS_DRIVER_FAILED_PRIOR_UNLOAD, -EIO,
	"STATUS_DRIVER_FAILED_PRIOR_UNLOAD"},
	{STATUS_SMARTCARD_SILENT_CONTEXT, -EIO,
	"STATUS_SMARTCARD_SILENT_CONTEXT"},
	{STATUS_PER_USER_TRUST_QUOTA_EXCEEDED, -EDQUOT,
	"STATUS_PER_USER_TRUST_QUOTA_EXCEEDED"},
	{STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED, -EDQUOT,
	"STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED"},
	{STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED, -EDQUOT,
	"STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED"},
	{STATUS_DS_NAME_NOT_UNIQUE, -EIO, "STATUS_DS_NAME_NOT_UNIQUE"},
	{STATUS_DS_DUPLICATE_ID_FOUND, -EIO, "STATUS_DS_DUPLICATE_ID_FOUND"},
	{STATUS_DS_GROUP_CONVERSION_ERROR, -EIO,
	"STATUS_DS_GROUP_CONVERSION_ERROR"},
	{STATUS_VOLSNAP_PREPARE_HIBERNATE, -EIO,
	"STATUS_VOLSNAP_PREPARE_HIBERNATE"},
	{STATUS_USER2USER_REQUIRED, -EIO, "STATUS_USER2USER_REQUIRED"},
	{STATUS_STACK_BUFFER_OVERRUN, -EIO, "STATUS_STACK_BUFFER_OVERRUN"},
	{STATUS_NO_S4U_PROT_SUPPORT, -EIO, "STATUS_NO_S4U_PROT_SUPPORT"},
	{STATUS_CROSSREALM_DELEGATION_FAILURE, -EIO,
	"STATUS_CROSSREALM_DELEGATION_FAILURE"},
	{STATUS_REVOCATION_OFFLINE_KDC, -EIO, "STATUS_REVOCATION_OFFLINE_KDC"},
	{STATUS_ISSUING_CA_UNTRUSTED_KDC, -EIO,
	"STATUS_ISSUING_CA_UNTRUSTED_KDC"},
	{STATUS_KDC_CERT_EXPIRED, -EIO, "STATUS_KDC_CERT_EXPIRED"},
	{STATUS_KDC_CERT_REVOKED, -EIO, "STATUS_KDC_CERT_REVOKED"},
	{STATUS_PARAMETER_QUOTA_EXCEEDED, -EDQUOT,
	"STATUS_PARAMETER_QUOTA_EXCEEDED"},
	{STATUS_HIBERNATION_FAILURE, -EIO, "STATUS_HIBERNATION_FAILURE"},
	{STATUS_DELAY_LOAD_FAILED, -EIO, "STATUS_DELAY_LOAD_FAILED"},
	{STATUS_AUTHENTICATION_FIREWALL_FAILED, -EIO,
	"STATUS_AUTHENTICATION_FIREWALL_FAILED"},
	{STATUS_VDM_DISALLOWED, -EIO, "STATUS_VDM_DISALLOWED"},
	{STATUS_HUNG_DISPLAY_DRIVER_THREAD, -EIO,
	"STATUS_HUNG_DISPLAY_DRIVER_THREAD"},
	{STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE, -EIO,
	"STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE"},
	{STATUS_INVALID_CRUNTIME_PARAMETER, -EIO,
	"STATUS_INVALID_CRUNTIME_PARAMETER"},
	{STATUS_NTLM_BLOCKED, -EIO, "STATUS_NTLM_BLOCKED"},
	{STATUS_ASSERTION_FAILURE, -EIO, "STATUS_ASSERTION_FAILURE"},
	{STATUS_VERIFIER_STOP, -EIO, "STATUS_VERIFIER_STOP"},
	{STATUS_CALLBACK_POP_STACK, -EIO, "STATUS_CALLBACK_POP_STACK"},
	{STATUS_INCOMPATIBLE_DRIVER_BLOCKED, -EIO,
	"STATUS_INCOMPATIBLE_DRIVER_BLOCKED"},
	{STATUS_HIVE_UNLOADED, -EIO, "STATUS_HIVE_UNLOADED"},
	{STATUS_COMPRESSION_DISABLED, -EIO, "STATUS_COMPRESSION_DISABLED"},
	{STATUS_FILE_SYSTEM_LIMITATION, -EIO, "STATUS_FILE_SYSTEM_LIMITATION"},
	{STATUS_INVALID_IMAGE_HASH, -EIO, "STATUS_INVALID_IMAGE_HASH"},
	{STATUS_NOT_CAPABLE, -EIO, "STATUS_NOT_CAPABLE"},
	{STATUS_REQUEST_OUT_OF_SEQUENCE, -EIO,
	"STATUS_REQUEST_OUT_OF_SEQUENCE"},
	{STATUS_IMPLEMENTATION_LIMIT, -EIO, "STATUS_IMPLEMENTATION_LIMIT"},
	{STATUS_ELEVATION_REQUIRED, -EIO, "STATUS_ELEVATION_REQUIRED"},
	{STATUS_BEYOND_VDL, -EIO, "STATUS_BEYOND_VDL"},
	{STATUS_ENCOUNTERED_WRITE_IN_PROGRESS, -EIO,
	"STATUS_ENCOUNTERED_WRITE_IN_PROGRESS"},
	{STATUS_PTE_CHANGED, -EIO, "STATUS_PTE_CHANGED"},
	{STATUS_PURGE_FAILED, -EIO, "STATUS_PURGE_FAILED"},
	{STATUS_CRED_REQUIRES_CONFIRMATION, -EIO,
	"STATUS_CRED_REQUIRES_CONFIRMATION"},
	{STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE, -EIO,
	"STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE"},
	{STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER, -EIO,
	"STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER"},
	{STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE, -EIO,
	"STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE"},
	{STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE, -EIO,
	"STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE"},
	{STATUS_CS_ENCRYPTION_FILE_NOT_CSE, -EIO,
	"STATUS_CS_ENCRYPTION_FILE_NOT_CSE"},
	{STATUS_INVALID_LABEL, -EIO, "STATUS_INVALID_LABEL"},
	{STATUS_DRIVER_PROCESS_TERMINATED, -EIO,
	"STATUS_DRIVER_PROCESS_TERMINATED"},
	{STATUS_AMBIGUOUS_SYSTEM_DEVICE, -EIO,
	"STATUS_AMBIGUOUS_SYSTEM_DEVICE"},
	{STATUS_SYSTEM_DEVICE_NOT_FOUND, -EIO,
	"STATUS_SYSTEM_DEVICE_NOT_FOUND"},
	{STATUS_RESTART_BOOT_APPLICATION, -EIO,
	"STATUS_RESTART_BOOT_APPLICATION"},
	{STATUS_INVALID_TASK_NAME, -EIO, "STATUS_INVALID_TASK_NAME"},
	{STATUS_INVALID_TASK_INDEX, -EIO, "STATUS_INVALID_TASK_INDEX"},
	{STATUS_THREAD_ALREADY_IN_TASK, -EIO, "STATUS_THREAD_ALREADY_IN_TASK"},
	{STATUS_CALLBACK_BYPASS, -EIO, "STATUS_CALLBACK_BYPASS"},
	{STATUS_PORT_CLOSED, -EIO, "STATUS_PORT_CLOSED"},
	{STATUS_MESSAGE_LOST, -EIO, "STATUS_MESSAGE_LOST"},
	{STATUS_INVALID_MESSAGE, -EIO, "STATUS_INVALID_MESSAGE"},
	{STATUS_REQUEST_CANCELED, -EIO, "STATUS_REQUEST_CANCELED"},
	{STATUS_RECURSIVE_DISPATCH, -EIO, "STATUS_RECURSIVE_DISPATCH"},
	{STATUS_LPC_RECEIVE_BUFFER_EXPECTED, -EIO,
	"STATUS_LPC_RECEIVE_BUFFER_EXPECTED"},
	{STATUS_LPC_INVALID_CONNECTION_USAGE, -EIO,
	"STATUS_LPC_INVALID_CONNECTION_USAGE"},
	{STATUS_LPC_REQUESTS_NOT_ALLOWED, -EIO,
	"STATUS_LPC_REQUESTS_NOT_ALLOWED"},
	{STATUS_RESOURCE_IN_USE, -EIO, "STATUS_RESOURCE_IN_USE"},
	{STATUS_HARDWARE_MEMORY_ERROR, -EIO, "STATUS_HARDWARE_MEMORY_ERROR"},
	{STATUS_THREADPOOL_HANDLE_EXCEPTION, -EIO,
	"STATUS_THREADPOOL_HANDLE_EXCEPTION"},
	{STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED, -EIO,
	"STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED"},
	{STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED, -EIO,
	"STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED"},
	{STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED, -EIO,
	"STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED"},
	{STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED, -EIO,
	"STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED"},
	{STATUS_THREADPOOL_RELEASED_DURING_OPERATION, -EIO,
	"STATUS_THREADPOOL_RELEASED_DURING_OPERATION"},
	{STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING, -EIO,
	"STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING"},
	{STATUS_APC_RETURNED_WHILE_IMPERSONATING, -EIO,
	"STATUS_APC_RETURNED_WHILE_IMPERSONATING"},
	{STATUS_PROCESS_IS_PROTECTED, -EIO, "STATUS_PROCESS_IS_PROTECTED"},
	{STATUS_MCA_EXCEPTION, -EIO, "STATUS_MCA_EXCEPTION"},
	{STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE, -EIO,
	"STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE"},
	{STATUS_SYMLINK_CLASS_DISABLED, -EIO, "STATUS_SYMLINK_CLASS_DISABLED"},
	{STATUS_INVALID_IDN_NORMALIZATION, -EIO,
	"STATUS_INVALID_IDN_NORMALIZATION"},
	{STATUS_NO_UNICODE_TRANSLATION, -EIO, "STATUS_NO_UNICODE_TRANSLATION"},
	{STATUS_ALREADY_REGISTERED, -EIO, "STATUS_ALREADY_REGISTERED"},
	{STATUS_CONTEXT_MISMATCH, -EIO, "STATUS_CONTEXT_MISMATCH"},
	{STATUS_PORT_ALREADY_HAS_COMPLETION_LIST, -EIO,
	"STATUS_PORT_ALREADY_HAS_COMPLETION_LIST"},
	{STATUS_CALLBACK_RETURNED_THREAD_PRIORITY, -EIO,
	"STATUS_CALLBACK_RETURNED_THREAD_PRIORITY"},
	{STATUS_INVALID_THREAD, -EIO, "STATUS_INVALID_THREAD"},
	{STATUS_CALLBACK_RETURNED_TRANSACTION, -EIO,
	"STATUS_CALLBACK_RETURNED_TRANSACTION"},
	{STATUS_CALLBACK_RETURNED_LDR_LOCK, -EIO,
	"STATUS_CALLBACK_RETURNED_LDR_LOCK"},
	{STATUS_CALLBACK_RETURNED_LANG, -EIO, "STATUS_CALLBACK_RETURNED_LANG"},
	{STATUS_CALLBACK_RETURNED_PRI_BACK, -EIO,
	"STATUS_CALLBACK_RETURNED_PRI_BACK"},
	{STATUS_CALLBACK_RETURNED_THREAD_AFFINITY, -EIO,
	"STATUS_CALLBACK_RETURNED_THREAD_AFFINITY"},
	{STATUS_DISK_REPAIR_DISABLED, -EIO, "STATUS_DISK_REPAIR_DISABLED"},
	{STATUS_DS_DOMAIN_RENAME_IN_PROGRESS, -EIO,
	"STATUS_DS_DOMAIN_RENAME_IN_PROGRESS"},
	{STATUS_DISK_QUOTA_EXCEEDED, -EDQUOT, "STATUS_DISK_QUOTA_EXCEEDED"},
	{STATUS_CONTENT_BLOCKED, -EIO, "STATUS_CONTENT_BLOCKED"},
	{STATUS_BAD_CLUSTERS, -EIO, "STATUS_BAD_CLUSTERS"},
	{STATUS_VOLUME_DIRTY, -EIO, "STATUS_VOLUME_DIRTY"},
	{STATUS_FILE_CHECKED_OUT, -EIO, "STATUS_FILE_CHECKED_OUT"},
	{STATUS_CHECKOUT_REQUIRED, -EIO, "STATUS_CHECKOUT_REQUIRED"},
	{STATUS_BAD_FILE_TYPE, -EIO, "STATUS_BAD_FILE_TYPE"},
	{STATUS_FILE_TOO_LARGE, -EIO, "STATUS_FILE_TOO_LARGE"},
	{STATUS_FORMS_AUTH_REQUIRED, -EIO, "STATUS_FORMS_AUTH_REQUIRED"},
	{STATUS_VIRUS_INFECTED, -EIO, "STATUS_VIRUS_INFECTED"},
	{STATUS_VIRUS_DELETED, -EIO, "STATUS_VIRUS_DELETED"},
	{STATUS_BAD_MCFG_TABLE, -EIO, "STATUS_BAD_MCFG_TABLE"},
	{STATUS_WOW_ASSERTION, -EIO, "STATUS_WOW_ASSERTION"},
	{STATUS_INVALID_SIGNATURE, -EIO, "STATUS_INVALID_SIGNATURE"},
	{STATUS_HMAC_NOT_SUPPORTED, -EIO, "STATUS_HMAC_NOT_SUPPORTED"},
	{STATUS_IPSEC_QUEUE_OVERFLOW, -EIO, "STATUS_IPSEC_QUEUE_OVERFLOW"},
	{STATUS_ND_QUEUE_OVERFLOW, -EIO, "STATUS_ND_QUEUE_OVERFLOW"},
	{STATUS_HOPLIMIT_EXCEEDED, -EIO, "STATUS_HOPLIMIT_EXCEEDED"},
	{STATUS_PROTOCOL_NOT_SUPPORTED, -EOPNOTSUPP,
	"STATUS_PROTOCOL_NOT_SUPPORTED"},
	{STATUS_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED, -EIO,
	"STATUS_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED"},
	{STATUS_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR, -EIO,
	"STATUS_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR"},
	{STATUS_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR, -EIO,
	"STATUS_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR"},
	{STATUS_XML_PARSE_ERROR, -EIO, "STATUS_XML_PARSE_ERROR"},
	{STATUS_XMLDSIG_ERROR, -EIO, "STATUS_XMLDSIG_ERROR"},
	{STATUS_WRONG_COMPARTMENT, -EIO, "STATUS_WRONG_COMPARTMENT"},
	{STATUS_AUTHIP_FAILURE, -EIO, "STATUS_AUTHIP_FAILURE"},
	{DBG_NO_STATE_CHANGE, -EIO, "DBG_NO_STATE_CHANGE"},
	{DBG_APP_NOT_IDLE, -EIO, "DBG_APP_NOT_IDLE"},
	{RPC_NT_INVALID_STRING_BINDING, -EIO, "RPC_NT_INVALID_STRING_BINDING"},
	{RPC_NT_WRONG_KIND_OF_BINDING, -EIO, "RPC_NT_WRONG_KIND_OF_BINDING"},
	{RPC_NT_INVALID_BINDING, -EIO, "RPC_NT_INVALID_BINDING"},
	{RPC_NT_PROTSEQ_NOT_SUPPORTED, -EOPNOTSUPP,
	"RPC_NT_PROTSEQ_NOT_SUPPORTED"},
	{RPC_NT_INVALID_RPC_PROTSEQ, -EIO, "RPC_NT_INVALID_RPC_PROTSEQ"},
	{RPC_NT_INVALID_STRING_UUID, -EIO, "RPC_NT_INVALID_STRING_UUID"},
	{RPC_NT_INVALID_ENDPOINT_FORMAT, -EIO,
	"RPC_NT_INVALID_ENDPOINT_FORMAT"},
	{RPC_NT_INVALID_NET_ADDR, -EIO, "RPC_NT_INVALID_NET_ADDR"},
	{RPC_NT_NO_ENDPOINT_FOUND, -EIO, "RPC_NT_NO_ENDPOINT_FOUND"},
	{RPC_NT_INVALID_TIMEOUT, -EINVAL, "RPC_NT_INVALID_TIMEOUT"},
	{RPC_NT_OBJECT_NOT_FOUND, -ENOENT, "RPC_NT_OBJECT_NOT_FOUND"},
	{RPC_NT_ALREADY_REGISTERED, -EIO, "RPC_NT_ALREADY_REGISTERED"},
	{RPC_NT_TYPE_ALREADY_REGISTERED, -EIO,
	"RPC_NT_TYPE_ALREADY_REGISTERED"},
	{RPC_NT_ALREADY_LISTENING, -EIO, "RPC_NT_ALREADY_LISTENING"},
	{RPC_NT_NO_PROTSEQS_REGISTERED, -EIO, "RPC_NT_NO_PROTSEQS_REGISTERED"},
	{RPC_NT_NOT_LISTENING, -EIO, "RPC_NT_NOT_LISTENING"},
	{RPC_NT_UNKNOWN_MGR_TYPE, -EIO, "RPC_NT_UNKNOWN_MGR_TYPE"},
	{RPC_NT_UNKNOWN_IF, -EIO, "RPC_NT_UNKNOWN_IF"},
	{RPC_NT_NO_BINDINGS, -EIO, "RPC_NT_NO_BINDINGS"},
	{RPC_NT_NO_PROTSEQS, -EIO, "RPC_NT_NO_PROTSEQS"},
	{RPC_NT_CANT_CREATE_ENDPOINT, -EIO, "RPC_NT_CANT_CREATE_ENDPOINT"},
	{RPC_NT_OUT_OF_RESOURCES, -EIO, "RPC_NT_OUT_OF_RESOURCES"},
	{RPC_NT_SERVER_UNAVAILABLE, -EIO, "RPC_NT_SERVER_UNAVAILABLE"},
	{RPC_NT_SERVER_TOO_BUSY, -EBUSY, "RPC_NT_SERVER_TOO_BUSY"},
	{RPC_NT_INVALID_NETWORK_OPTIONS, -EIO,
	"RPC_NT_INVALID_NETWORK_OPTIONS"},
	{RPC_NT_NO_CALL_ACTIVE, -EIO, "RPC_NT_NO_CALL_ACTIVE"},
	{RPC_NT_CALL_FAILED, -EIO, "RPC_NT_CALL_FAILED"},
	{RPC_NT_CALL_FAILED_DNE, -EIO, "RPC_NT_CALL_FAILED_DNE"},
	{RPC_NT_PROTOCOL_ERROR, -EIO, "RPC_NT_PROTOCOL_ERROR"},
	{RPC_NT_UNSUPPORTED_TRANS_SYN, -EIO, "RPC_NT_UNSUPPORTED_TRANS_SYN"},
	{RPC_NT_UNSUPPORTED_TYPE, -EIO, "RPC_NT_UNSUPPORTED_TYPE"},
	{RPC_NT_INVALID_TAG, -EIO, "RPC_NT_INVALID_TAG"},
	{RPC_NT_INVALID_BOUND, -EIO, "RPC_NT_INVALID_BOUND"},
	{RPC_NT_NO_ENTRY_NAME, -EIO, "RPC_NT_NO_ENTRY_NAME"},
	{RPC_NT_INVALID_NAME_SYNTAX, -EIO, "RPC_NT_INVALID_NAME_SYNTAX"},
	{RPC_NT_UNSUPPORTED_NAME_SYNTAX, -EIO,
	"RPC_NT_UNSUPPORTED_NAME_SYNTAX"},
	{RPC_NT_UUID_NO_ADDRESS, -EIO, "RPC_NT_UUID_NO_ADDRESS"},
	{RPC_NT_DUPLICATE_ENDPOINT, -ENOTUNIQ, "RPC_NT_DUPLICATE_ENDPOINT"},
	{RPC_NT_UNKNOWN_AUTHN_TYPE, -EIO, "RPC_NT_UNKNOWN_AUTHN_TYPE"},
	{RPC_NT_MAX_CALLS_TOO_SMALL, -EIO, "RPC_NT_MAX_CALLS_TOO_SMALL"},
	{RPC_NT_STRING_TOO_LONG, -EIO, "RPC_NT_STRING_TOO_LONG"},
	{RPC_NT_PROTSEQ_NOT_FOUND, -EIO, "RPC_NT_PROTSEQ_NOT_FOUND"},
	{RPC_NT_PROCNUM_OUT_OF_RANGE, -EIO, "RPC_NT_PROCNUM_OUT_OF_RANGE"},
	{RPC_NT_BINDING_HAS_NO_AUTH, -EIO, "RPC_NT_BINDING_HAS_NO_AUTH"},
	{RPC_NT_UNKNOWN_AUTHN_SERVICE, -EIO, "RPC_NT_UNKNOWN_AUTHN_SERVICE"},
	{RPC_NT_UNKNOWN_AUTHN_LEVEL, -EIO, "RPC_NT_UNKNOWN_AUTHN_LEVEL"},
	{RPC_NT_INVALID_AUTH_IDENTITY, -EIO, "RPC_NT_INVALID_AUTH_IDENTITY"},
	{RPC_NT_UNKNOWN_AUTHZ_SERVICE, -EIO, "RPC_NT_UNKNOWN_AUTHZ_SERVICE"},
	{EPT_NT_INVALID_ENTRY, -EIO, "EPT_NT_INVALID_ENTRY"},
	{EPT_NT_CANT_PERFORM_OP, -EIO, "EPT_NT_CANT_PERFORM_OP"},
	{EPT_NT_NOT_REGISTERED, -EIO, "EPT_NT_NOT_REGISTERED"},
	{RPC_NT_NOTHING_TO_EXPORT, -EIO, "RPC_NT_NOTHING_TO_EXPORT"},
	{RPC_NT_INCOMPLETE_NAME, -EIO, "RPC_NT_INCOMPLETE_NAME"},
	{RPC_NT_INVALID_VERS_OPTION, -EIO, "RPC_NT_INVALID_VERS_OPTION"},
	{RPC_NT_NO_MORE_MEMBERS, -EIO, "RPC_NT_NO_MORE_MEMBERS"},
	{RPC_NT_NOT_ALL_OBJS_UNEXPORTED, -EIO,
	"RPC_NT_NOT_ALL_OBJS_UNEXPORTED"},
	{RPC_NT_INTERFACE_NOT_FOUND, -EIO, "RPC_NT_INTERFACE_NOT_FOUND"},
	{RPC_NT_ENTRY_ALREADY_EXISTS, -EIO, "RPC_NT_ENTRY_ALREADY_EXISTS"},
	{RPC_NT_ENTRY_NOT_FOUND, -EIO, "RPC_NT_ENTRY_NOT_FOUND"},
	{RPC_NT_NAME_SERVICE_UNAVAILABLE, -EIO,
	"RPC_NT_NAME_SERVICE_UNAVAILABLE"},
	{RPC_NT_INVALID_NAF_ID, -EIO, "RPC_NT_INVALID_NAF_ID"},
	{RPC_NT_CANNOT_SUPPORT, -EOPNOTSUPP, "RPC_NT_CANNOT_SUPPORT"},
	{RPC_NT_NO_CONTEXT_AVAILABLE, -EIO, "RPC_NT_NO_CONTEXT_AVAILABLE"},
	{RPC_NT_INTERNAL_ERROR, -EIO, "RPC_NT_INTERNAL_ERROR"},
	{RPC_NT_ZERO_DIVIDE, -EIO, "RPC_NT_ZERO_DIVIDE"},
	{RPC_NT_ADDRESS_ERROR, -EIO, "RPC_NT_ADDRESS_ERROR"},
	{RPC_NT_FP_DIV_ZERO, -EIO, "RPC_NT_FP_DIV_ZERO"},
	{RPC_NT_FP_UNDERFLOW, -EIO, "RPC_NT_FP_UNDERFLOW"},
	{RPC_NT_FP_OVERFLOW, -EIO, "RPC_NT_FP_OVERFLOW"},
	{RPC_NT_CALL_IN_PROGRESS, -EIO, "RPC_NT_CALL_IN_PROGRESS"},
	{RPC_NT_NO_MORE_BINDINGS, -EIO, "RPC_NT_NO_MORE_BINDINGS"},
	{RPC_NT_GROUP_MEMBER_NOT_FOUND, -EIO, "RPC_NT_GROUP_MEMBER_NOT_FOUND"},
	{EPT_NT_CANT_CREATE, -EIO, "EPT_NT_CANT_CREATE"},
	{RPC_NT_INVALID_OBJECT, -EIO, "RPC_NT_INVALID_OBJECT"},
	{RPC_NT_NO_INTERFACES, -EIO, "RPC_NT_NO_INTERFACES"},
	{RPC_NT_CALL_CANCELLED, -EIO, "RPC_NT_CALL_CANCELLED"},
	{RPC_NT_BINDING_INCOMPLETE, -EIO, "RPC_NT_BINDING_INCOMPLETE"},
	{RPC_NT_COMM_FAILURE, -EIO, "RPC_NT_COMM_FAILURE"},
	{RPC_NT_UNSUPPORTED_AUTHN_LEVEL, -EIO,
	"RPC_NT_UNSUPPORTED_AUTHN_LEVEL"},
	{RPC_NT_NO_PRINC_NAME, -EIO, "RPC_NT_NO_PRINC_NAME"},
	{RPC_NT_NOT_RPC_ERROR, -EIO, "RPC_NT_NOT_RPC_ERROR"},
	{RPC_NT_SEC_PKG_ERROR, -EIO, "RPC_NT_SEC_PKG_ERROR"},
	{RPC_NT_NOT_CANCELLED, -EIO, "RPC_NT_NOT_CANCELLED"},
	{RPC_NT_INVALID_ASYNC_HANDLE, -EIO, "RPC_NT_INVALID_ASYNC_HANDLE"},
	{RPC_NT_INVALID_ASYNC_CALL, -EIO, "RPC_NT_INVALID_ASYNC_CALL"},
	{RPC_NT_PROXY_ACCESS_DENIED, -EACCES, "RPC_NT_PROXY_ACCESS_DENIED"},
	{RPC_NT_NO_MORE_ENTRIES, -EIO, "RPC_NT_NO_MORE_ENTRIES"},
	{RPC_NT_SS_CHAR_TRANS_OPEN_FAIL, -EIO,
	"RPC_NT_SS_CHAR_TRANS_OPEN_FAIL"},
	{RPC_NT_SS_CHAR_TRANS_SHORT_FILE, -EIO,
	"RPC_NT_SS_CHAR_TRANS_SHORT_FILE"},
	{RPC_NT_SS_IN_NULL_CONTEXT, -EIO, "RPC_NT_SS_IN_NULL_CONTEXT"},
	{RPC_NT_SS_CONTEXT_MISMATCH, -EIO, "RPC_NT_SS_CONTEXT_MISMATCH"},
	{RPC_NT_SS_CONTEXT_DAMAGED, -EIO, "RPC_NT_SS_CONTEXT_DAMAGED"},
	{RPC_NT_SS_HANDLES_MISMATCH, -EIO, "RPC_NT_SS_HANDLES_MISMATCH"},
	{RPC_NT_SS_CANNOT_GET_CALL_HANDLE, -EIO,
	"RPC_NT_SS_CANNOT_GET_CALL_HANDLE"},
	{RPC_NT_NULL_REF_POINTER, -EIO, "RPC_NT_NULL_REF_POINTER"},
	{RPC_NT_ENUM_VALUE_OUT_OF_RANGE, -EIO,
	"RPC_NT_ENUM_VALUE_OUT_OF_RANGE"},
	{RPC_NT_BYTE_COUNT_TOO_SMALL, -EIO, "RPC_NT_BYTE_COUNT_TOO_SMALL"},
	{RPC_NT_BAD_STUB_DATA, -EIO, "RPC_NT_BAD_STUB_DATA"},
	{RPC_NT_INVALID_ES_ACTION, -EIO, "RPC_NT_INVALID_ES_ACTION"},
	{RPC_NT_WRONG_ES_VERSION, -EIO, "RPC_NT_WRONG_ES_VERSION"},
	{RPC_NT_WRONG_STUB_VERSION, -EIO, "RPC_NT_WRONG_STUB_VERSION"},
	{RPC_NT_INVALID_PIPE_OBJECT, -EIO, "RPC_NT_INVALID_PIPE_OBJECT"},
	{RPC_NT_INVALID_PIPE_OPERATION, -EIO, "RPC_NT_INVALID_PIPE_OPERATION"},
	{RPC_NT_WRONG_PIPE_VERSION, -EIO, "RPC_NT_WRONG_PIPE_VERSION"},
	{RPC_NT_PIPE_CLOSED, -EIO, "RPC_NT_PIPE_CLOSED"},
	{RPC_NT_PIPE_DISCIPLINE_ERROR, -EIO, "RPC_NT_PIPE_DISCIPLINE_ERROR"},
	{RPC_NT_PIPE_EMPTY, -EIO, "RPC_NT_PIPE_EMPTY"},
	{STATUS_PNP_BAD_MPS_TABLE, -EIO, "STATUS_PNP_BAD_MPS_TABLE"},
	{STATUS_PNP_TRANSLATION_FAILED, -EIO, "STATUS_PNP_TRANSLATION_FAILED"},
	{STATUS_PNP_IRQ_TRANSLATION_FAILED, -EIO,
	"STATUS_PNP_IRQ_TRANSLATION_FAILED"},
	{STATUS_PNP_INVALID_ID, -EIO, "STATUS_PNP_INVALID_ID"},
	{STATUS_IO_REISSUE_AS_CACHED, -EIO, "STATUS_IO_REISSUE_AS_CACHED"},
	{STATUS_CTX_WINSTATION_NAME_INVALID, -EIO,
	"STATUS_CTX_WINSTATION_NAME_INVALID"},
	{STATUS_CTX_INVALID_PD, -EIO, "STATUS_CTX_INVALID_PD"},
	{STATUS_CTX_PD_NOT_FOUND, -EIO, "STATUS_CTX_PD_NOT_FOUND"},
	{STATUS_CTX_CLOSE_PENDING, -EIO, "STATUS_CTX_CLOSE_PENDING"},
	{STATUS_CTX_NO_OUTBUF, -EIO, "STATUS_CTX_NO_OUTBUF"},
	{STATUS_CTX_MODEM_INF_NOT_FOUND, -EIO,
	"STATUS_CTX_MODEM_INF_NOT_FOUND"},
	{STATUS_CTX_INVALID_MODEMNAME, -EIO, "STATUS_CTX_INVALID_MODEMNAME"},
	{STATUS_CTX_RESPONSE_ERROR, -EIO, "STATUS_CTX_RESPONSE_ERROR"},
	{STATUS_CTX_MODEM_RESPONSE_TIMEOUT, -ETIMEDOUT,
	"STATUS_CTX_MODEM_RESPONSE_TIMEOUT"},
	{STATUS_CTX_MODEM_RESPONSE_NO_CARRIER, -EIO,
	"STATUS_CTX_MODEM_RESPONSE_NO_CARRIER"},
	{STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE, -EIO,
	"STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE"},
	{STATUS_CTX_MODEM_RESPONSE_BUSY, -EBUSY,
	"STATUS_CTX_MODEM_RESPONSE_BUSY"},
	{STATUS_CTX_MODEM_RESPONSE_VOICE, -EIO,
	"STATUS_CTX_MODEM_RESPONSE_VOICE"},
	{STATUS_CTX_TD_ERROR, -EIO, "STATUS_CTX_TD_ERROR"},
	{STATUS_CTX_LICENSE_CLIENT_INVALID, -EIO,
	"STATUS_CTX_LICENSE_CLIENT_INVALID"},
	{STATUS_CTX_LICENSE_NOT_AVAILABLE, -EIO,
	"STATUS_CTX_LICENSE_NOT_AVAILABLE"},
	{STATUS_CTX_LICENSE_EXPIRED, -EIO, "STATUS_CTX_LICENSE_EXPIRED"},
	{STATUS_CTX_WINSTATION_NOT_FOUND, -EIO,
	"STATUS_CTX_WINSTATION_NOT_FOUND"},
	{STATUS_CTX_WINSTATION_NAME_COLLISION, -EIO,
	"STATUS_CTX_WINSTATION_NAME_COLLISION"},
	{STATUS_CTX_WINSTATION_BUSY, -EBUSY, "STATUS_CTX_WINSTATION_BUSY"},
	{STATUS_CTX_BAD_VIDEO_MODE, -EIO, "STATUS_CTX_BAD_VIDEO_MODE"},
	{STATUS_CTX_GRAPHICS_INVALID, -EIO, "STATUS_CTX_GRAPHICS_INVALID"},
	{STATUS_CTX_NOT_CONSOLE, -EIO, "STATUS_CTX_NOT_CONSOLE"},
	{STATUS_CTX_CLIENT_QUERY_TIMEOUT, -EIO,
	"STATUS_CTX_CLIENT_QUERY_TIMEOUT"},
	{STATUS_CTX_CONSOLE_DISCONNECT, -EIO, "STATUS_CTX_CONSOLE_DISCONNECT"},
	{STATUS_CTX_CONSOLE_CONNECT, -EIO, "STATUS_CTX_CONSOLE_CONNECT"},
	{STATUS_CTX_SHADOW_DENIED, -EIO, "STATUS_CTX_SHADOW_DENIED"},
	{STATUS_CTX_WINSTATION_ACCESS_DENIED, -EACCES,
	"STATUS_CTX_WINSTATION_ACCESS_DENIED"},
	{STATUS_CTX_INVALID_WD, -EIO, "STATUS_CTX_INVALID_WD"},
	{STATUS_CTX_WD_NOT_FOUND, -EIO, "STATUS_CTX_WD_NOT_FOUND"},
	{STATUS_CTX_SHADOW_INVALID, -EIO, "STATUS_CTX_SHADOW_INVALID"},
	{STATUS_CTX_SHADOW_DISABLED, -EIO, "STATUS_CTX_SHADOW_DISABLED"},
	{STATUS_RDP_PROTOCOL_ERROR, -EIO, "STATUS_RDP_PROTOCOL_ERROR"},
	{STATUS_CTX_CLIENT_LICENSE_NOT_SET, -EIO,
	"STATUS_CTX_CLIENT_LICENSE_NOT_SET"},
	{STATUS_CTX_CLIENT_LICENSE_IN_USE, -EIO,
	"STATUS_CTX_CLIENT_LICENSE_IN_USE"},
	{STATUS_CTX_SHADOW_ENDED_BY_MODE_CHANGE, -EIO,
	"STATUS_CTX_SHADOW_ENDED_BY_MODE_CHANGE"},
	{STATUS_CTX_SHADOW_NOT_RUNNING, -EIO, "STATUS_CTX_SHADOW_NOT_RUNNING"},
	{STATUS_CTX_LOGON_DISABLED, -EIO, "STATUS_CTX_LOGON_DISABLED"},
	{STATUS_CTX_SECURITY_LAYER_ERROR, -EIO,
	"STATUS_CTX_SECURITY_LAYER_ERROR"},
	{STATUS_TS_INCOMPATIBLE_SESSIONS, -EIO,
	"STATUS_TS_INCOMPATIBLE_SESSIONS"},
	{STATUS_MUI_FILE_NOT_FOUND, -EIO, "STATUS_MUI_FILE_NOT_FOUND"},
	{STATUS_MUI_INVALID_FILE, -EIO, "STATUS_MUI_INVALID_FILE"},
	{STATUS_MUI_INVALID_RC_CONFIG, -EIO, "STATUS_MUI_INVALID_RC_CONFIG"},
	{STATUS_MUI_INVALID_LOCALE_NAME, -EIO,
	"STATUS_MUI_INVALID_LOCALE_NAME"},
	{STATUS_MUI_INVALID_ULTIMATEFALLBACK_NAME, -EIO,
	"STATUS_MUI_INVALID_ULTIMATEFALLBACK_NAME"},
	{STATUS_MUI_FILE_NOT_LOADED, -EIO, "STATUS_MUI_FILE_NOT_LOADED"},
	{STATUS_RESOURCE_ENUM_USER_STOP, -EIO,
	"STATUS_RESOURCE_ENUM_USER_STOP"},
	{STATUS_CLUSTER_INVALID_NODE, -EIO, "STATUS_CLUSTER_INVALID_NODE"},
	{STATUS_CLUSTER_NODE_EXISTS, -EIO, "STATUS_CLUSTER_NODE_EXISTS"},
	{STATUS_CLUSTER_JOIN_IN_PROGRESS, -EIO,
	"STATUS_CLUSTER_JOIN_IN_PROGRESS"},
	{STATUS_CLUSTER_NODE_NOT_FOUND, -EIO, "STATUS_CLUSTER_NODE_NOT_FOUND"},
	{STATUS_CLUSTER_LOCAL_NODE_NOT_FOUND, -EIO,
	"STATUS_CLUSTER_LOCAL_NODE_NOT_FOUND"},
	{STATUS_CLUSTER_NETWORK_EXISTS, -EIO, "STATUS_CLUSTER_NETWORK_EXISTS"},
	{STATUS_CLUSTER_NETWORK_NOT_FOUND, -EIO,
	"STATUS_CLUSTER_NETWORK_NOT_FOUND"},
	{STATUS_CLUSTER_NETINTERFACE_EXISTS, -EIO,
	"STATUS_CLUSTER_NETINTERFACE_EXISTS"},
	{STATUS_CLUSTER_NETINTERFACE_NOT_FOUND, -EIO,
	"STATUS_CLUSTER_NETINTERFACE_NOT_FOUND"},
	{STATUS_CLUSTER_INVALID_REQUEST, -EIO,
	"STATUS_CLUSTER_INVALID_REQUEST"},
	{STATUS_CLUSTER_INVALID_NETWORK_PROVIDER, -EIO,
	"STATUS_CLUSTER_INVALID_NETWORK_PROVIDER"},
	{STATUS_CLUSTER_NODE_DOWN, -EIO, "STATUS_CLUSTER_NODE_DOWN"},
	{STATUS_CLUSTER_NODE_UNREACHABLE, -EIO,
	"STATUS_CLUSTER_NODE_UNREACHABLE"},
	{STATUS_CLUSTER_NODE_NOT_MEMBER, -EIO,
	"STATUS_CLUSTER_NODE_NOT_MEMBER"},
	{STATUS_CLUSTER_JOIN_NOT_IN_PROGRESS, -EIO,
	"STATUS_CLUSTER_JOIN_NOT_IN_PROGRESS"},
	{STATUS_CLUSTER_INVALID_NETWORK, -EIO,
	"STATUS_CLUSTER_INVALID_NETWORK"},
	{STATUS_CLUSTER_NO_NET_ADAPTERS, -EIO,
	"STATUS_CLUSTER_NO_NET_ADAPTERS"},
	{STATUS_CLUSTER_NODE_UP, -EIO, "STATUS_CLUSTER_NODE_UP"},
	{STATUS_CLUSTER_NODE_PAUSED, -EIO, "STATUS_CLUSTER_NODE_PAUSED"},
	{STATUS_CLUSTER_NODE_NOT_PAUSED, -EIO,
	"STATUS_CLUSTER_NODE_NOT_PAUSED"},
	{STATUS_CLUSTER_NO_SECURITY_CONTEXT, -EIO,
	"STATUS_CLUSTER_NO_SECURITY_CONTEXT"},
	{STATUS_CLUSTER_NETWORK_NOT_INTERNAL, -EIO,
	"STATUS_CLUSTER_NETWORK_NOT_INTERNAL"},
	{STATUS_CLUSTER_POISONED, -EIO, "STATUS_CLUSTER_POISONED"},
	{STATUS_ACPI_INVALID_OPCODE, -EIO, "STATUS_ACPI_INVALID_OPCODE"},
	{STATUS_ACPI_STACK_OVERFLOW, -EIO, "STATUS_ACPI_STACK_OVERFLOW"},
	{STATUS_ACPI_ASSERT_FAILED, -EIO, "STATUS_ACPI_ASSERT_FAILED"},
	{STATUS_ACPI_INVALID_INDEX, -EIO, "STATUS_ACPI_INVALID_INDEX"},
	{STATUS_ACPI_INVALID_ARGUMENT, -EIO, "STATUS_ACPI_INVALID_ARGUMENT"},
	{STATUS_ACPI_FATAL, -EIO, "STATUS_ACPI_FATAL"},
	{STATUS_ACPI_INVALID_SUPERNAME, -EIO, "STATUS_ACPI_INVALID_SUPERNAME"},
	{STATUS_ACPI_INVALID_ARGTYPE, -EIO, "STATUS_ACPI_INVALID_ARGTYPE"},
	{STATUS_ACPI_INVALID_OBJTYPE, -EIO, "STATUS_ACPI_INVALID_OBJTYPE"},
	{STATUS_ACPI_INVALID_TARGETTYPE, -EIO,
	"STATUS_ACPI_INVALID_TARGETTYPE"},
	{STATUS_ACPI_INCORRECT_ARGUMENT_COUNT, -EIO,
	"STATUS_ACPI_INCORRECT_ARGUMENT_COUNT"},
	{STATUS_ACPI_ADDRESS_NOT_MAPPED, -EIO,
	"STATUS_ACPI_ADDRESS_NOT_MAPPED"},
	{STATUS_ACPI_INVALID_EVENTTYPE, -EIO, "STATUS_ACPI_INVALID_EVENTTYPE"},
	{STATUS_ACPI_HANDLER_COLLISION, -EIO, "STATUS_ACPI_HANDLER_COLLISION"},
	{STATUS_ACPI_INVALID_DATA, -EIO, "STATUS_ACPI_INVALID_DATA"},
	{STATUS_ACPI_INVALID_REGION, -EIO, "STATUS_ACPI_INVALID_REGION"},
	{STATUS_ACPI_INVALID_ACCESS_SIZE, -EIO,
	"STATUS_ACPI_INVALID_ACCESS_SIZE"},
	{STATUS_ACPI_ACQUIRE_GLOBAL_LOCK, -EIO,
	"STATUS_ACPI_ACQUIRE_GLOBAL_LOCK"},
	{STATUS_ACPI_ALREADY_INITIALIZED, -EIO,
	"STATUS_ACPI_ALREADY_INITIALIZED"},
	{STATUS_ACPI_NOT_INITIALIZED, -EIO, "STATUS_ACPI_NOT_INITIALIZED"},
	{STATUS_ACPI_INVALID_MUTEX_LEVEL, -EIO,
	"STATUS_ACPI_INVALID_MUTEX_LEVEL"},
	{STATUS_ACPI_MUTEX_NOT_OWNED, -EIO, "STATUS_ACPI_MUTEX_NOT_OWNED"},
	{STATUS_ACPI_MUTEX_NOT_OWNER, -EIO, "STATUS_ACPI_MUTEX_NOT_OWNER"},
	{STATUS_ACPI_RS_ACCESS, -EIO, "STATUS_ACPI_RS_ACCESS"},
	{STATUS_ACPI_INVALID_TABLE, -EIO, "STATUS_ACPI_INVALID_TABLE"},
	{STATUS_ACPI_REG_HANDLER_FAILED, -EIO,
	"STATUS_ACPI_REG_HANDLER_FAILED"},
	{STATUS_ACPI_POWER_REQUEST_FAILED, -EIO,
	"STATUS_ACPI_POWER_REQUEST_FAILED"},
	{STATUS_SXS_SECTION_NOT_FOUND, -EIO, "STATUS_SXS_SECTION_NOT_FOUND"},
	{STATUS_SXS_CANT_GEN_ACTCTX, -EIO, "STATUS_SXS_CANT_GEN_ACTCTX"},
	{STATUS_SXS_INVALID_ACTCTXDATA_FORMAT, -EIO,
	"STATUS_SXS_INVALID_ACTCTXDATA_FORMAT"},
	{STATUS_SXS_ASSEMBLY_NOT_FOUND, -EIO, "STATUS_SXS_ASSEMBLY_NOT_FOUND"},
	{STATUS_SXS_MANIFEST_FORMAT_ERROR, -EIO,
	"STATUS_SXS_MANIFEST_FORMAT_ERROR"},
	{STATUS_SXS_MANIFEST_PARSE_ERROR, -EIO,
	"STATUS_SXS_MANIFEST_PARSE_ERROR"},
	{STATUS_SXS_ACTIVATION_CONTEXT_DISABLED, -EIO,
	"STATUS_SXS_ACTIVATION_CONTEXT_DISABLED"},
	{STATUS_SXS_KEY_NOT_FOUND, -EIO, "STATUS_SXS_KEY_NOT_FOUND"},
	{STATUS_SXS_VERSION_CONFLICT, -EIO, "STATUS_SXS_VERSION_CONFLICT"},
	{STATUS_SXS_WRONG_SECTION_TYPE, -EIO, "STATUS_SXS_WRONG_SECTION_TYPE"},
	{STATUS_SXS_THREAD_QUERIES_DISABLED, -EIO,
	"STATUS_SXS_THREAD_QUERIES_DISABLED"},
	{STATUS_SXS_ASSEMBLY_MISSING, -EIO, "STATUS_SXS_ASSEMBLY_MISSING"},
	{STATUS_SXS_PROCESS_DEFAULT_ALREADY_SET, -EIO,
	"STATUS_SXS_PROCESS_DEFAULT_ALREADY_SET"},
	{STATUS_SXS_EARLY_DEACTIVATION, -EIO, "STATUS_SXS_EARLY_DEACTIVATION"},
	{STATUS_SXS_INVALID_DEACTIVATION, -EIO,
	"STATUS_SXS_INVALID_DEACTIVATION"},
	{STATUS_SXS_MULTIPLE_DEACTIVATION, -EIO,
	"STATUS_SXS_MULTIPLE_DEACTIVATION"},
	{STATUS_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY, -EIO,
	"STATUS_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY"},
	{STATUS_SXS_PROCESS_TERMINATION_REQUESTED, -EIO,
	"STATUS_SXS_PROCESS_TERMINATION_REQUESTED"},
	{STATUS_SXS_CORRUPT_ACTIVATION_STACK, -EIO,
	"STATUS_SXS_CORRUPT_ACTIVATION_STACK"},
	{STATUS_SXS_CORRUPTION, -EIO, "STATUS_SXS_CORRUPTION"},
	{STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE, -EIO,
	"STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE"},
	{STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME, -EIO,
	"STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME"},
	{STATUS_SXS_IDENTITY_DUPLICATE_ATTRIBUTE, -EIO,
	"STATUS_SXS_IDENTITY_DUPLICATE_ATTRIBUTE"},
	{STATUS_SXS_IDENTITY_PARSE_ERROR, -EIO,
	"STATUS_SXS_IDENTITY_PARSE_ERROR"},
	{STATUS_SXS_COMPONENT_STORE_CORRUPT, -EIO,
	"STATUS_SXS_COMPONENT_STORE_CORRUPT"},
	{STATUS_SXS_FILE_HASH_MISMATCH, -EIO, "STATUS_SXS_FILE_HASH_MISMATCH"},
	{STATUS_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT, -EIO,
	"STATUS_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT"},
	{STATUS_SXS_IDENTITIES_DIFFERENT, -EIO,
	"STATUS_SXS_IDENTITIES_DIFFERENT"},
	{STATUS_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT, -EIO,
	"STATUS_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT"},
	{STATUS_SXS_FILE_NOT_PART_OF_ASSEMBLY, -EIO,
	"STATUS_SXS_FILE_NOT_PART_OF_ASSEMBLY"},
	{STATUS_ADVANCED_INSTALLER_FAILED, -EIO,
	"STATUS_ADVANCED_INSTALLER_FAILED"},
	{STATUS_XML_ENCODING_MISMATCH, -EIO, "STATUS_XML_ENCODING_MISMATCH"},
	{STATUS_SXS_MANIFEST_TOO_BIG, -EIO, "STATUS_SXS_MANIFEST_TOO_BIG"},
	{STATUS_SXS_SETTING_NOT_REGISTERED, -EIO,
	"STATUS_SXS_SETTING_NOT_REGISTERED"},
	{STATUS_SXS_TRANSACTION_CLOSURE_INCOMPLETE, -EIO,
	"STATUS_SXS_TRANSACTION_CLOSURE_INCOMPLETE"},
	{STATUS_SMI_PRIMITIVE_INSTALLER_FAILED, -EIO,
	"STATUS_SMI_PRIMITIVE_INSTALLER_FAILED"},
	{STATUS_GENERIC_COMMAND_FAILED, -EIO, "STATUS_GENERIC_COMMAND_FAILED"},
	{STATUS_SXS_FILE_HASH_MISSING, -EIO, "STATUS_SXS_FILE_HASH_MISSING"},
	{STATUS_TRANSACTIONAL_CONFLICT, -EIO, "STATUS_TRANSACTIONAL_CONFLICT"},
	{STATUS_INVALID_TRANSACTION, -EIO, "STATUS_INVALID_TRANSACTION"},
	{STATUS_TRANSACTION_NOT_ACTIVE, -EIO, "STATUS_TRANSACTION_NOT_ACTIVE"},
	{STATUS_TM_INITIALIZATION_FAILED, -EIO,
	"STATUS_TM_INITIALIZATION_FAILED"},
	{STATUS_RM_NOT_ACTIVE, -EIO, "STATUS_RM_NOT_ACTIVE"},
	{STATUS_RM_METADATA_CORRUPT, -EIO, "STATUS_RM_METADATA_CORRUPT"},
	{STATUS_TRANSACTION_NOT_JOINED, -EIO, "STATUS_TRANSACTION_NOT_JOINED"},
	{STATUS_DIRECTORY_NOT_RM, -EIO, "STATUS_DIRECTORY_NOT_RM"},
	{STATUS_TRANSACTIONS_UNSUPPORTED_REMOTE, -EIO,
	"STATUS_TRANSACTIONS_UNSUPPORTED_REMOTE"},
	{STATUS_LOG_RESIZE_INVALID_SIZE, -EIO,
	"STATUS_LOG_RESIZE_INVALID_SIZE"},
	{STATUS_REMOTE_FILE_VERSION_MISMATCH, -EIO,
	"STATUS_REMOTE_FILE_VERSION_MISMATCH"},
	{STATUS_CRM_PROTOCOL_ALREADY_EXISTS, -EIO,
	"STATUS_CRM_PROTOCOL_ALREADY_EXISTS"},
	{STATUS_TRANSACTION_PROPAGATION_FAILED, -EIO,
	"STATUS_TRANSACTION_PROPAGATION_FAILED"},
	{STATUS_CRM_PROTOCOL_NOT_FOUND, -EIO, "STATUS_CRM_PROTOCOL_NOT_FOUND"},
	{STATUS_TRANSACTION_SUPERIOR_EXISTS, -EIO,
	"STATUS_TRANSACTION_SUPERIOR_EXISTS"},
	{STATUS_TRANSACTION_REQUEST_NOT_VALID, -EIO,
	"STATUS_TRANSACTION_REQUEST_NOT_VALID"},
	{STATUS_TRANSACTION_NOT_REQUESTED, -EIO,
	"STATUS_TRANSACTION_NOT_REQUESTED"},
	{STATUS_TRANSACTION_ALREADY_ABORTED, -EIO,
	"STATUS_TRANSACTION_ALREADY_ABORTED"},
	{STATUS_TRANSACTION_ALREADY_COMMITTED, -EIO,
	"STATUS_TRANSACTION_ALREADY_COMMITTED"},
	{STATUS_TRANSACTION_INVALID_MARSHALL_BUFFER, -EIO,
	"STATUS_TRANSACTION_INVALID_MARSHALL_BUFFER"},
	{STATUS_CURRENT_TRANSACTION_NOT_VALID, -EIO,
	"STATUS_CURRENT_TRANSACTION_NOT_VALID"},
	{STATUS_LOG_GROWTH_FAILED, -EIO, "STATUS_LOG_GROWTH_FAILED"},
	{STATUS_OBJECT_NO_LONGER_EXISTS, -EIO,
	"STATUS_OBJECT_NO_LONGER_EXISTS"},
	{STATUS_STREAM_MINIVERSION_NOT_FOUND, -EIO,
	"STATUS_STREAM_MINIVERSION_NOT_FOUND"},
	{STATUS_STREAM_MINIVERSION_NOT_VALID, -EIO,
	"STATUS_STREAM_MINIVERSION_NOT_VALID"},
	{STATUS_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION, -EIO,
	"STATUS_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION"},
	{STATUS_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT, -EIO,
	"STATUS_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT"},
	{STATUS_CANT_CREATE_MORE_STREAM_MINIVERSIONS, -EIO,
	"STATUS_CANT_CREATE_MORE_STREAM_MINIVERSIONS"},
	{STATUS_HANDLE_NO_LONGER_VALID, -EIO, "STATUS_HANDLE_NO_LONGER_VALID"},
	{STATUS_LOG_CORRUPTION_DETECTED, -EIO,
	"STATUS_LOG_CORRUPTION_DETECTED"},
	{STATUS_RM_DISCONNECTED, -EIO, "STATUS_RM_DISCONNECTED"},
	{STATUS_ENLISTMENT_NOT_SUPERIOR, -EIO,
	"STATUS_ENLISTMENT_NOT_SUPERIOR"},
	{STATUS_FILE_IDENTITY_NOT_PERSISTENT, -EIO,
	"STATUS_FILE_IDENTITY_NOT_PERSISTENT"},
	{STATUS_CANT_BREAK_TRANSACTIONAL_DEPENDENCY, -EIO,
	"STATUS_CANT_BREAK_TRANSACTIONAL_DEPENDENCY"},
	{STATUS_CANT_CROSS_RM_BOUNDARY, -EIO, "STATUS_CANT_CROSS_RM_BOUNDARY"},
	{STATUS_TXF_DIR_NOT_EMPTY, -EIO, "STATUS_TXF_DIR_NOT_EMPTY"},
	{STATUS_INDOUBT_TRANSACTIONS_EXIST, -EIO,
	"STATUS_INDOUBT_TRANSACTIONS_EXIST"},
	{STATUS_TM_VOLATILE, -EIO, "STATUS_TM_VOLATILE"},
	{STATUS_ROLLBACK_TIMER_EXPIRED, -EIO, "STATUS_ROLLBACK_TIMER_EXPIRED"},
	{STATUS_TXF_ATTRIBUTE_CORRUPT, -EIO, "STATUS_TXF_ATTRIBUTE_CORRUPT"},
	{STATUS_EFS_NOT_ALLOWED_IN_TRANSACTION, -EIO,
	"STATUS_EFS_NOT_ALLOWED_IN_TRANSACTION"},
	{STATUS_TRANSACTIONAL_OPEN_NOT_ALLOWED, -EIO,
	"STATUS_TRANSACTIONAL_OPEN_NOT_ALLOWED"},
	{STATUS_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE, -EIO,
	"STATUS_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE"},
	{STATUS_TRANSACTION_REQUIRED_PROMOTION, -EIO,
	"STATUS_TRANSACTION_REQUIRED_PROMOTION"},
	{STATUS_CANNOT_EXECUTE_FILE_IN_TRANSACTION, -EIO,
	"STATUS_CANNOT_EXECUTE_FILE_IN_TRANSACTION"},
	{STATUS_TRANSACTIONS_NOT_FROZEN, -EIO,
	"STATUS_TRANSACTIONS_NOT_FROZEN"},
	{STATUS_TRANSACTION_FREEZE_IN_PROGRESS, -EIO,
	"STATUS_TRANSACTION_FREEZE_IN_PROGRESS"},
	{STATUS_NOT_SNAPSHOT_VOLUME, -EIO, "STATUS_NOT_SNAPSHOT_VOLUME"},
	{STATUS_NO_SAVEPOINT_WITH_OPEN_FILES, -EIO,
	"STATUS_NO_SAVEPOINT_WITH_OPEN_FILES"},
	{STATUS_SPARSE_NOT_ALLOWED_IN_TRANSACTION, -EIO,
	"STATUS_SPARSE_NOT_ALLOWED_IN_TRANSACTION"},
	{STATUS_TM_IDENTITY_MISMATCH, -EIO, "STATUS_TM_IDENTITY_MISMATCH"},
	{STATUS_FLOATED_SECTION, -EIO, "STATUS_FLOATED_SECTION"},
	{STATUS_CANNOT_ACCEPT_TRANSACTED_WORK, -EIO,
	"STATUS_CANNOT_ACCEPT_TRANSACTED_WORK"},
	{STATUS_CANNOT_ABORT_TRANSACTIONS, -EIO,
	"STATUS_CANNOT_ABORT_TRANSACTIONS"},
	{STATUS_TRANSACTION_NOT_FOUND, -EIO, "STATUS_TRANSACTION_NOT_FOUND"},
	{STATUS_RESOURCEMANAGER_NOT_FOUND, -EIO,
	"STATUS_RESOURCEMANAGER_NOT_FOUND"},
	{STATUS_ENLISTMENT_NOT_FOUND, -EIO, "STATUS_ENLISTMENT_NOT_FOUND"},
	{STATUS_TRANSACTIONMANAGER_NOT_FOUND, -EIO,
	"STATUS_TRANSACTIONMANAGER_NOT_FOUND"},
	{STATUS_TRANSACTIONMANAGER_NOT_ONLINE, -EIO,
	"STATUS_TRANSACTIONMANAGER_NOT_ONLINE"},
	{STATUS_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION, -EIO,
	"STATUS_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION"},
	{STATUS_TRANSACTION_NOT_ROOT, -EIO, "STATUS_TRANSACTION_NOT_ROOT"},
	{STATUS_TRANSACTION_OBJECT_EXPIRED, -EIO,
	"STATUS_TRANSACTION_OBJECT_EXPIRED"},
	{STATUS_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION, -EIO,
	"STATUS_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION"},
	{STATUS_TRANSACTION_RESPONSE_NOT_ENLISTED, -EIO,
	"STATUS_TRANSACTION_RESPONSE_NOT_ENLISTED"},
	{STATUS_TRANSACTION_RECORD_TOO_LONG, -EIO,
	"STATUS_TRANSACTION_RECORD_TOO_LONG"},
	{STATUS_NO_LINK_TRACKING_IN_TRANSACTION, -EIO,
	"STATUS_NO_LINK_TRACKING_IN_TRANSACTION"},
	{STATUS_OPERATION_NOT_SUPPORTED_IN_TRANSACTION, -EOPNOTSUPP,
	"STATUS_OPERATION_NOT_SUPPORTED_IN_TRANSACTION"},
	{STATUS_TRANSACTION_INTEGRITY_VIOLATED, -EIO,
	"STATUS_TRANSACTION_INTEGRITY_VIOLATED"},
	{STATUS_LOG_SECTOR_INVALID, -EIO, "STATUS_LOG_SECTOR_INVALID"},
	{STATUS_LOG_SECTOR_PARITY_INVALID, -EIO,
	"STATUS_LOG_SECTOR_PARITY_INVALID"},
	{STATUS_LOG_SECTOR_REMAPPED, -EIO, "STATUS_LOG_SECTOR_REMAPPED"},
	{STATUS_LOG_BLOCK_INCOMPLETE, -EIO, "STATUS_LOG_BLOCK_INCOMPLETE"},
	{STATUS_LOG_INVALID_RANGE, -EIO, "STATUS_LOG_INVALID_RANGE"},
	{STATUS_LOG_BLOCKS_EXHAUSTED, -EIO, "STATUS_LOG_BLOCKS_EXHAUSTED"},
	{STATUS_LOG_READ_CONTEXT_INVALID, -EIO,
	"STATUS_LOG_READ_CONTEXT_INVALID"},
	{STATUS_LOG_RESTART_INVALID, -EIO, "STATUS_LOG_RESTART_INVALID"},
	{STATUS_LOG_BLOCK_VERSION, -EIO, "STATUS_LOG_BLOCK_VERSION"},
	{STATUS_LOG_BLOCK_INVALID, -EIO, "STATUS_LOG_BLOCK_INVALID"},
	{STATUS_LOG_READ_MODE_INVALID, -EIO, "STATUS_LOG_READ_MODE_INVALID"},
	{STATUS_LOG_METADATA_CORRUPT, -EIO, "STATUS_LOG_METADATA_CORRUPT"},
	{STATUS_LOG_METADATA_INVALID, -EIO, "STATUS_LOG_METADATA_INVALID"},
	{STATUS_LOG_METADATA_INCONSISTENT, -EIO,
	"STATUS_LOG_METADATA_INCONSISTENT"},
	{STATUS_LOG_RESERVATION_INVALID, -EIO,
	"STATUS_LOG_RESERVATION_INVALID"},
	{STATUS_LOG_CANT_DELETE, -EIO, "STATUS_LOG_CANT_DELETE"},
	{STATUS_LOG_CONTAINER_LIMIT_EXCEEDED, -EIO,
	"STATUS_LOG_CONTAINER_LIMIT_EXCEEDED"},
	{STATUS_LOG_START_OF_LOG, -EIO, "STATUS_LOG_START_OF_LOG"},
	{STATUS_LOG_POLICY_ALREADY_INSTALLED, -EIO,
	"STATUS_LOG_POLICY_ALREADY_INSTALLED"},
	{STATUS_LOG_POLICY_NOT_INSTALLED, -EIO,
	"STATUS_LOG_POLICY_NOT_INSTALLED"},
	{STATUS_LOG_POLICY_INVALID, -EIO, "STATUS_LOG_POLICY_INVALID"},
	{STATUS_LOG_POLICY_CONFLICT, -EIO, "STATUS_LOG_POLICY_CONFLICT"},
	{STATUS_LOG_PINNED_ARCHIVE_TAIL, -EIO,
	"STATUS_LOG_PINNED_ARCHIVE_TAIL"},
	{STATUS_LOG_RECORD_NONEXISTENT, -EIO, "STATUS_LOG_RECORD_NONEXISTENT"},
	{STATUS_LOG_RECORDS_RESERVED_INVALID, -EIO,
	"STATUS_LOG_RECORDS_RESERVED_INVALID"},
	{STATUS_LOG_SPACE_RESERVED_INVALID, -EIO,
	"STATUS_LOG_SPACE_RESERVED_INVALID"},
	{STATUS_LOG_TAIL_INVALID, -EIO, "STATUS_LOG_TAIL_INVALID"},
	{STATUS_LOG_FULL, -EIO, "STATUS_LOG_FULL"},
	{STATUS_LOG_MULTIPLEXED, -EIO, "STATUS_LOG_MULTIPLEXED"},
	{STATUS_LOG_DEDICATED, -EIO, "STATUS_LOG_DEDICATED"},
	{STATUS_LOG_ARCHIVE_NOT_IN_PROGRESS, -EIO,
	"STATUS_LOG_ARCHIVE_NOT_IN_PROGRESS"},
	{STATUS_LOG_ARCHIVE_IN_PROGRESS, -EIO,
	"STATUS_LOG_ARCHIVE_IN_PROGRESS"},
	{STATUS_LOG_EPHEMERAL, -EIO, "STATUS_LOG_EPHEMERAL"},
	{STATUS_LOG_NOT_ENOUGH_CONTAINERS, -EIO,
	"STATUS_LOG_NOT_ENOUGH_CONTAINERS"},
	{STATUS_LOG_CLIENT_ALREADY_REGISTERED, -EIO,
	"STATUS_LOG_CLIENT_ALREADY_REGISTERED"},
	{STATUS_LOG_CLIENT_NOT_REGISTERED, -EIO,
	"STATUS_LOG_CLIENT_NOT_REGISTERED"},
	{STATUS_LOG_FULL_HANDLER_IN_PROGRESS, -EIO,
	"STATUS_LOG_FULL_HANDLER_IN_PROGRESS"},
	{STATUS_LOG_CONTAINER_READ_FAILED, -EIO,
	"STATUS_LOG_CONTAINER_READ_FAILED"},
	{STATUS_LOG_CONTAINER_WRITE_FAILED, -EIO,
	"STATUS_LOG_CONTAINER_WRITE_FAILED"},
	{STATUS_LOG_CONTAINER_OPEN_FAILED, -EIO,
	"STATUS_LOG_CONTAINER_OPEN_FAILED"},
	{STATUS_LOG_CONTAINER_STATE_INVALID, -EIO,
	"STATUS_LOG_CONTAINER_STATE_INVALID"},
	{STATUS_LOG_STATE_INVALID, -EIO, "STATUS_LOG_STATE_INVALID"},
	{STATUS_LOG_PINNED, -EIO, "STATUS_LOG_PINNED"},
	{STATUS_LOG_METADATA_FLUSH_FAILED, -EIO,
	"STATUS_LOG_METADATA_FLUSH_FAILED"},
	{STATUS_LOG_INCONSISTENT_SECURITY, -EIO,
	"STATUS_LOG_INCONSISTENT_SECURITY"},
	{STATUS_LOG_APPENDED_FLUSH_FAILED, -EIO,
	"STATUS_LOG_APPENDED_FLUSH_FAILED"},
	{STATUS_LOG_PINNED_RESERVATION, -EIO, "STATUS_LOG_PINNED_RESERVATION"},
	{STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD, -EIO,
	"STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD"},
	{STATUS_FLT_NO_HANDLER_DEFINED, -EIO, "STATUS_FLT_NO_HANDLER_DEFINED"},
	{STATUS_FLT_CONTEXT_ALREADY_DEFINED, -EIO,
	"STATUS_FLT_CONTEXT_ALREADY_DEFINED"},
	{STATUS_FLT_INVALID_ASYNCHRONOUS_REQUEST, -EIO,
	"STATUS_FLT_INVALID_ASYNCHRONOUS_REQUEST"},
	{STATUS_FLT_DISALLOW_FAST_IO, -EIO, "STATUS_FLT_DISALLOW_FAST_IO"},
	{STATUS_FLT_INVALID_NAME_REQUEST, -EIO,
	"STATUS_FLT_INVALID_NAME_REQUEST"},
	{STATUS_FLT_NOT_SAFE_TO_POST_OPERATION, -EIO,
	"STATUS_FLT_NOT_SAFE_TO_POST_OPERATION"},
	{STATUS_FLT_NOT_INITIALIZED, -EIO, "STATUS_FLT_NOT_INITIALIZED"},
	{STATUS_FLT_FILTER_NOT_READY, -EIO, "STATUS_FLT_FILTER_NOT_READY"},
	{STATUS_FLT_POST_OPERATION_CLEANUP, -EIO,
	"STATUS_FLT_POST_OPERATION_CLEANUP"},
	{STATUS_FLT_INTERNAL_ERROR, -EIO, "STATUS_FLT_INTERNAL_ERROR"},
	{STATUS_FLT_DELETING_OBJECT, -EIO, "STATUS_FLT_DELETING_OBJECT"},
	{STATUS_FLT_MUST_BE_NONPAGED_POOL, -EIO,
	"STATUS_FLT_MUST_BE_NONPAGED_POOL"},
	{STATUS_FLT_DUPLICATE_ENTRY, -EIO, "STATUS_FLT_DUPLICATE_ENTRY"},
	{STATUS_FLT_CBDQ_DISABLED, -EIO, "STATUS_FLT_CBDQ_DISABLED"},
	{STATUS_FLT_DO_NOT_ATTACH, -EIO, "STATUS_FLT_DO_NOT_ATTACH"},
	{STATUS_FLT_DO_NOT_DETACH, -EIO, "STATUS_FLT_DO_NOT_DETACH"},
	{STATUS_FLT_INSTANCE_ALTITUDE_COLLISION, -EIO,
	"STATUS_FLT_INSTANCE_ALTITUDE_COLLISION"},
	{STATUS_FLT_INSTANCE_NAME_COLLISION, -EIO,
	"STATUS_FLT_INSTANCE_NAME_COLLISION"},
	{STATUS_FLT_FILTER_NOT_FOUND, -EIO, "STATUS_FLT_FILTER_NOT_FOUND"},
	{STATUS_FLT_VOLUME_NOT_FOUND, -EIO, "STATUS_FLT_VOLUME_NOT_FOUND"},
	{STATUS_FLT_INSTANCE_NOT_FOUND, -EIO, "STATUS_FLT_INSTANCE_NOT_FOUND"},
	{STATUS_FLT_CONTEXT_ALLOCATION_NOT_FOUND, -EIO,
	"STATUS_FLT_CONTEXT_ALLOCATION_NOT_FOUND"},
	{STATUS_FLT_INVALID_CONTEXT_REGISTRATION, -EIO,
	"STATUS_FLT_INVALID_CONTEXT_REGISTRATION"},
	{STATUS_FLT_NAME_CACHE_MISS, -EIO, "STATUS_FLT_NAME_CACHE_MISS"},
	{STATUS_FLT_NO_DEVICE_OBJECT, -EIO, "STATUS_FLT_NO_DEVICE_OBJECT"},
	{STATUS_FLT_VOLUME_ALREADY_MOUNTED, -EIO,
	"STATUS_FLT_VOLUME_ALREADY_MOUNTED"},
	{STATUS_FLT_ALREADY_ENLISTED, -EIO, "STATUS_FLT_ALREADY_ENLISTED"},
	{STATUS_FLT_CONTEXT_ALREADY_LINKED, -EIO,
	"STATUS_FLT_CONTEXT_ALREADY_LINKED"},
	{STATUS_FLT_NO_WAITER_FOR_REPLY, -EIO,
	"STATUS_FLT_NO_WAITER_FOR_REPLY"},
	{STATUS_MONITOR_NO_DESCRIPTOR, -EIO, "STATUS_MONITOR_NO_DESCRIPTOR"},
	{STATUS_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT, -EIO,
	"STATUS_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT"},
	{STATUS_MONITOR_INVALID_DESCRIPTOR_CHECKSUM, -EIO,
	"STATUS_MONITOR_INVALID_DESCRIPTOR_CHECKSUM"},
	{STATUS_MONITOR_INVALID_STANDARD_TIMING_BLOCK, -EIO,
	"STATUS_MONITOR_INVALID_STANDARD_TIMING_BLOCK"},
	{STATUS_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED, -EIO,
	"STATUS_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED"},
	{STATUS_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK, -EIO,
	"STATUS_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK"},
	{STATUS_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK, -EIO,
	"STATUS_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK"},
	{STATUS_MONITOR_NO_MORE_DESCRIPTOR_DATA, -EIO,
	"STATUS_MONITOR_NO_MORE_DESCRIPTOR_DATA"},
	{STATUS_MONITOR_INVALID_DETAILED_TIMING_BLOCK, -EIO,
	"STATUS_MONITOR_INVALID_DETAILED_TIMING_BLOCK"},
	{STATUS_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER, -EIO,
	"STATUS_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER"},
	{STATUS_GRAPHICS_INSUFFICIENT_DMA_BUFFER, -EIO,
	"STATUS_GRAPHICS_INSUFFICIENT_DMA_BUFFER"},
	{STATUS_GRAPHICS_INVALID_DISPLAY_ADAPTER, -EIO,
	"STATUS_GRAPHICS_INVALID_DISPLAY_ADAPTER"},
	{STATUS_GRAPHICS_ADAPTER_WAS_RESET, -EIO,
	"STATUS_GRAPHICS_ADAPTER_WAS_RESET"},
	{STATUS_GRAPHICS_INVALID_DRIVER_MODEL, -EIO,
	"STATUS_GRAPHICS_INVALID_DRIVER_MODEL"},
	{STATUS_GRAPHICS_PRESENT_MODE_CHANGED, -EIO,
	"STATUS_GRAPHICS_PRESENT_MODE_CHANGED"},
	{STATUS_GRAPHICS_PRESENT_OCCLUDED, -EIO,
	"STATUS_GRAPHICS_PRESENT_OCCLUDED"},
	{STATUS_GRAPHICS_PRESENT_DENIED, -EIO,
	"STATUS_GRAPHICS_PRESENT_DENIED"},
	{STATUS_GRAPHICS_CANNOTCOLORCONVERT, -EIO,
	"STATUS_GRAPHICS_CANNOTCOLORCONVERT"},
	{STATUS_GRAPHICS_NO_VIDEO_MEMORY, -EIO,
	"STATUS_GRAPHICS_NO_VIDEO_MEMORY"},
	{STATUS_GRAPHICS_CANT_LOCK_MEMORY, -EIO,
	"STATUS_GRAPHICS_CANT_LOCK_MEMORY"},
	{STATUS_GRAPHICS_ALLOCATION_BUSY, -EBUSY,
	"STATUS_GRAPHICS_ALLOCATION_BUSY"},
	{STATUS_GRAPHICS_TOO_MANY_REFERENCES, -EIO,
	"STATUS_GRAPHICS_TOO_MANY_REFERENCES"},
	{STATUS_GRAPHICS_TRY_AGAIN_LATER, -EIO,
	"STATUS_GRAPHICS_TRY_AGAIN_LATER"},
	{STATUS_GRAPHICS_TRY_AGAIN_NOW, -EIO, "STATUS_GRAPHICS_TRY_AGAIN_NOW"},
	{STATUS_GRAPHICS_ALLOCATION_INVALID, -EIO,
	"STATUS_GRAPHICS_ALLOCATION_INVALID"},
	{STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE, -EIO,
	"STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE"},
	{STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED, -EIO,
	"STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED"},
	{STATUS_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION, -EIO,
	"STATUS_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION"},
	{STATUS_GRAPHICS_INVALID_ALLOCATION_USAGE, -EIO,
	"STATUS_GRAPHICS_INVALID_ALLOCATION_USAGE"},
	{STATUS_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION, -EIO,
	"STATUS_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION"},
	{STATUS_GRAPHICS_ALLOCATION_CLOSED, -EIO,
	"STATUS_GRAPHICS_ALLOCATION_CLOSED"},
	{STATUS_GRAPHICS_INVALID_ALLOCATION_INSTANCE, -EIO,
	"STATUS_GRAPHICS_INVALID_ALLOCATION_INSTANCE"},
	{STATUS_GRAPHICS_INVALID_ALLOCATION_HANDLE, -EIO,
	"STATUS_GRAPHICS_INVALID_ALLOCATION_HANDLE"},
	{STATUS_GRAPHICS_WRONG_ALLOCATION_DEVICE, -EIO,
	"STATUS_GRAPHICS_WRONG_ALLOCATION_DEVICE"},
	{STATUS_GRAPHICS_ALLOCATION_CONTENT_LOST, -EIO,
	"STATUS_GRAPHICS_ALLOCATION_CONTENT_LOST"},
	{STATUS_GRAPHICS_GPU_EXCEPTION_ON_DEVICE, -EIO,
	"STATUS_GRAPHICS_GPU_EXCEPTION_ON_DEVICE"},
	{STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY"},
	{STATUS_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_INVALID_VIDPN, -EIO, "STATUS_GRAPHICS_INVALID_VIDPN"},
	{STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE"},
	{STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET"},
	{STATUS_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_INVALID_VIDPN_SOURCEMODESET, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDPN_SOURCEMODESET"},
	{STATUS_GRAPHICS_INVALID_VIDPN_TARGETMODESET, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDPN_TARGETMODESET"},
	{STATUS_GRAPHICS_INVALID_FREQUENCY, -EIO,
	"STATUS_GRAPHICS_INVALID_FREQUENCY"},
	{STATUS_GRAPHICS_INVALID_ACTIVE_REGION, -EIO,
	"STATUS_GRAPHICS_INVALID_ACTIVE_REGION"},
	{STATUS_GRAPHICS_INVALID_TOTAL_REGION, -EIO,
	"STATUS_GRAPHICS_INVALID_TOTAL_REGION"},
	{STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE"},
	{STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE"},
	{STATUS_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET, -EIO,
	"STATUS_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET"},
	{STATUS_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY, -EIO,
	"STATUS_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY"},
	{STATUS_GRAPHICS_MODE_ALREADY_IN_MODESET, -EIO,
	"STATUS_GRAPHICS_MODE_ALREADY_IN_MODESET"},
	{STATUS_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET"},
	{STATUS_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET"},
	{STATUS_GRAPHICS_SOURCE_ALREADY_IN_SET, -EIO,
	"STATUS_GRAPHICS_SOURCE_ALREADY_IN_SET"},
	{STATUS_GRAPHICS_TARGET_ALREADY_IN_SET, -EIO,
	"STATUS_GRAPHICS_TARGET_ALREADY_IN_SET"},
	{STATUS_GRAPHICS_INVALID_VIDPN_PRESENT_PATH, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDPN_PRESENT_PATH"},
	{STATUS_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY, -EIO,
	"STATUS_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY"},
	{STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET, -EIO,
	"STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET"},
	{STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE, -EIO,
	"STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE"},
	{STATUS_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET, -EIO,
	"STATUS_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET"},
	{STATUS_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET, -EIO,
	"STATUS_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET"},
	{STATUS_GRAPHICS_STALE_MODESET, -EIO, "STATUS_GRAPHICS_STALE_MODESET"},
	{STATUS_GRAPHICS_INVALID_MONITOR_SOURCEMODESET, -EIO,
	"STATUS_GRAPHICS_INVALID_MONITOR_SOURCEMODESET"},
	{STATUS_GRAPHICS_INVALID_MONITOR_SOURCE_MODE, -EIO,
	"STATUS_GRAPHICS_INVALID_MONITOR_SOURCE_MODE"},
	{STATUS_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN, -EIO,
	"STATUS_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN"},
	{STATUS_GRAPHICS_MODE_ID_MUST_BE_UNIQUE, -EIO,
	"STATUS_GRAPHICS_MODE_ID_MUST_BE_UNIQUE"},
	{STATUS_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION, -EIO,
	"STATUS_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION"},
	{STATUS_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES, -EIO,
	"STATUS_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES"},
	{STATUS_GRAPHICS_PATH_NOT_IN_TOPOLOGY, -EIO,
	"STATUS_GRAPHICS_PATH_NOT_IN_TOPOLOGY"},
	{STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE, -EIO,
	"STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE"},
	{STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET, -EIO,
	"STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET"},
	{STATUS_GRAPHICS_INVALID_MONITORDESCRIPTORSET, -EIO,
	"STATUS_GRAPHICS_INVALID_MONITORDESCRIPTORSET"},
	{STATUS_GRAPHICS_INVALID_MONITORDESCRIPTOR, -EIO,
	"STATUS_GRAPHICS_INVALID_MONITORDESCRIPTOR"},
	{STATUS_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET, -EIO,
	"STATUS_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET"},
	{STATUS_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET, -EIO,
	"STATUS_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET"},
	{STATUS_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE, -EIO,
	"STATUS_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE"},
	{STATUS_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE"},
	{STATUS_GRAPHICS_RESOURCES_NOT_RELATED, -EIO,
	"STATUS_GRAPHICS_RESOURCES_NOT_RELATED"},
	{STATUS_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE, -EIO,
	"STATUS_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE"},
	{STATUS_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE, -EIO,
	"STATUS_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE"},
	{STATUS_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET, -EIO,
	"STATUS_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET"},
	{STATUS_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER, -EIO,
	"STATUS_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER"},
	{STATUS_GRAPHICS_NO_VIDPNMGR, -EIO, "STATUS_GRAPHICS_NO_VIDPNMGR"},
	{STATUS_GRAPHICS_NO_ACTIVE_VIDPN, -EIO,
	"STATUS_GRAPHICS_NO_ACTIVE_VIDPN"},
	{STATUS_GRAPHICS_STALE_VIDPN_TOPOLOGY, -EIO,
	"STATUS_GRAPHICS_STALE_VIDPN_TOPOLOGY"},
	{STATUS_GRAPHICS_MONITOR_NOT_CONNECTED, -EIO,
	"STATUS_GRAPHICS_MONITOR_NOT_CONNECTED"},
	{STATUS_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY, -EIO,
	"STATUS_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY"},
	{STATUS_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE, -EIO,
	"STATUS_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE"},
	{STATUS_GRAPHICS_INVALID_VISIBLEREGION_SIZE, -EIO,
	"STATUS_GRAPHICS_INVALID_VISIBLEREGION_SIZE"},
	{STATUS_GRAPHICS_INVALID_STRIDE, -EIO,
	"STATUS_GRAPHICS_INVALID_STRIDE"},
	{STATUS_GRAPHICS_INVALID_PIXELFORMAT, -EIO,
	"STATUS_GRAPHICS_INVALID_PIXELFORMAT"},
	{STATUS_GRAPHICS_INVALID_COLORBASIS, -EIO,
	"STATUS_GRAPHICS_INVALID_COLORBASIS"},
	{STATUS_GRAPHICS_INVALID_PIXELVALUEACCESSMODE, -EIO,
	"STATUS_GRAPHICS_INVALID_PIXELVALUEACCESSMODE"},
	{STATUS_GRAPHICS_TARGET_NOT_IN_TOPOLOGY, -EIO,
	"STATUS_GRAPHICS_TARGET_NOT_IN_TOPOLOGY"},
	{STATUS_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT, -EIO,
	"STATUS_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT"},
	{STATUS_GRAPHICS_VIDPN_SOURCE_IN_USE, -EIO,
	"STATUS_GRAPHICS_VIDPN_SOURCE_IN_USE"},
	{STATUS_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN, -EIO,
	"STATUS_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN"},
	{STATUS_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL, -EIO,
	"STATUS_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL"},
	{STATUS_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION, -EIO,
	"STATUS_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION"},
	{STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED,
	-EIO,
	"STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_INVALID_GAMMA_RAMP, -EIO,
	"STATUS_GRAPHICS_INVALID_GAMMA_RAMP"},
	{STATUS_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_MODE_NOT_IN_MODESET, -EIO,
	"STATUS_GRAPHICS_MODE_NOT_IN_MODESET"},
	{STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON, -EIO,
	"STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON"},
	{STATUS_GRAPHICS_INVALID_PATH_CONTENT_TYPE, -EIO,
	"STATUS_GRAPHICS_INVALID_PATH_CONTENT_TYPE"},
	{STATUS_GRAPHICS_INVALID_COPYPROTECTION_TYPE, -EIO,
	"STATUS_GRAPHICS_INVALID_COPYPROTECTION_TYPE"},
	{STATUS_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS, -EIO,
	"STATUS_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS"},
	{STATUS_GRAPHICS_INVALID_SCANLINE_ORDERING, -EIO,
	"STATUS_GRAPHICS_INVALID_SCANLINE_ORDERING"},
	{STATUS_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED, -EIO,
	"STATUS_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED"},
	{STATUS_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS, -EIO,
	"STATUS_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS"},
	{STATUS_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT, -EIO,
	"STATUS_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT"},
	{STATUS_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM, -EIO,
	"STATUS_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM"},
	{STATUS_GRAPHICS_INVALID_MONITOR_CAPABILITY_ORIGIN, -EIO,
	"STATUS_GRAPHICS_INVALID_MONITOR_CAPABILITY_ORIGIN"},
	{STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE_CONSTRAINT, -EIO,
	"STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE_CONSTRAINT"},
	{STATUS_GRAPHICS_MAX_NUM_PATHS_REACHED, -EIO,
	"STATUS_GRAPHICS_MAX_NUM_PATHS_REACHED"},
	{STATUS_GRAPHICS_CANCEL_VIDPN_TOPOLOGY_AUGMENTATION, -EIO,
	"STATUS_GRAPHICS_CANCEL_VIDPN_TOPOLOGY_AUGMENTATION"},
	{STATUS_GRAPHICS_INVALID_CLIENT_TYPE, -EIO,
	"STATUS_GRAPHICS_INVALID_CLIENT_TYPE"},
	{STATUS_GRAPHICS_CLIENTVIDPN_NOT_SET, -EIO,
	"STATUS_GRAPHICS_CLIENTVIDPN_NOT_SET"},
	{STATUS_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED, -EIO,
	"STATUS_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED"},
	{STATUS_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_NOT_A_LINKED_ADAPTER, -EIO,
	"STATUS_GRAPHICS_NOT_A_LINKED_ADAPTER"},
	{STATUS_GRAPHICS_LEADLINK_NOT_ENUMERATED, -EIO,
	"STATUS_GRAPHICS_LEADLINK_NOT_ENUMERATED"},
	{STATUS_GRAPHICS_CHAINLINKS_NOT_ENUMERATED, -EIO,
	"STATUS_GRAPHICS_CHAINLINKS_NOT_ENUMERATED"},
	{STATUS_GRAPHICS_ADAPTER_CHAIN_NOT_READY, -EIO,
	"STATUS_GRAPHICS_ADAPTER_CHAIN_NOT_READY"},
	{STATUS_GRAPHICS_CHAINLINKS_NOT_STARTED, -EIO,
	"STATUS_GRAPHICS_CHAINLINKS_NOT_STARTED"},
	{STATUS_GRAPHICS_CHAINLINKS_NOT_POWERED_ON, -EIO,
	"STATUS_GRAPHICS_CHAINLINKS_NOT_POWERED_ON"},
	{STATUS_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE, -EIO,
	"STATUS_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE"},
	{STATUS_GRAPHICS_NOT_POST_DEVICE_DRIVER, -EIO,
	"STATUS_GRAPHICS_NOT_POST_DEVICE_DRIVER"},
	{STATUS_GRAPHICS_ADAPTER_ACCESS_NOT_EXCLUDED, -EIO,
	"STATUS_GRAPHICS_ADAPTER_ACCESS_NOT_EXCLUDED"},
	{STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_COPP_SEMANTICS,
	-EIO,
	"STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_COPP_SEMANTICS"},
	{STATUS_GRAPHICS_OPM_INVALID_INFORMATION_REQUEST, -EIO,
	"STATUS_GRAPHICS_OPM_INVALID_INFORMATION_REQUEST"},
	{STATUS_GRAPHICS_OPM_DRIVER_INTERNAL_ERROR, -EIO,
	"STATUS_GRAPHICS_OPM_DRIVER_INTERNAL_ERROR"},
	{STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_OPM_SEMANTICS, -EIO,
	"STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_OPM_SEMANTICS"},
	{STATUS_GRAPHICS_OPM_SIGNALING_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_OPM_SIGNALING_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_OPM_INVALID_CONFIGURATION_REQUEST, -EIO,
	"STATUS_GRAPHICS_OPM_INVALID_CONFIGURATION_REQUEST"},
	{STATUS_GRAPHICS_OPM_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_OPM_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_COPP_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_COPP_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_UAB_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_UAB_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS, -EIO,
	"STATUS_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS"},
	{STATUS_GRAPHICS_OPM_PARAMETER_ARRAY_TOO_SMALL, -EIO,
	"STATUS_GRAPHICS_OPM_PARAMETER_ARRAY_TOO_SMALL"},
	{STATUS_GRAPHICS_OPM_NO_PROTECTED_OUTPUTS_EXIST, -EIO,
	"STATUS_GRAPHICS_OPM_NO_PROTECTED_OUTPUTS_EXIST"},
	{STATUS_GRAPHICS_PVP_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME, -EIO,
	"STATUS_GRAPHICS_PVP_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME"},
	{STATUS_GRAPHICS_PVP_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP, -EIO,
	"STATUS_GRAPHICS_PVP_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP"},
	{STATUS_GRAPHICS_PVP_MIRRORING_DEVICES_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_PVP_MIRRORING_DEVICES_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_OPM_INVALID_POINTER, -EIO,
	"STATUS_GRAPHICS_OPM_INVALID_POINTER"},
	{STATUS_GRAPHICS_OPM_INTERNAL_ERROR, -EIO,
	"STATUS_GRAPHICS_OPM_INTERNAL_ERROR"},
	{STATUS_GRAPHICS_OPM_INVALID_HANDLE, -EIO,
	"STATUS_GRAPHICS_OPM_INVALID_HANDLE"},
	{STATUS_GRAPHICS_PVP_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE, -EIO,
	"STATUS_GRAPHICS_PVP_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE"},
	{STATUS_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH, -EIO,
	"STATUS_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH"},
	{STATUS_GRAPHICS_OPM_SPANNING_MODE_ENABLED, -EIO,
	"STATUS_GRAPHICS_OPM_SPANNING_MODE_ENABLED"},
	{STATUS_GRAPHICS_OPM_THEATER_MODE_ENABLED, -EIO,
	"STATUS_GRAPHICS_OPM_THEATER_MODE_ENABLED"},
	{STATUS_GRAPHICS_PVP_HFS_FAILED, -EIO,
	"STATUS_GRAPHICS_PVP_HFS_FAILED"},
	{STATUS_GRAPHICS_OPM_INVALID_SRM, -EIO,
	"STATUS_GRAPHICS_OPM_INVALID_SRM"},
	{STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP, -EIO,
	"STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP"},
	{STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP, -EIO,
	"STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP"},
	{STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA, -EIO,
	"STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA"},
	{STATUS_GRAPHICS_OPM_HDCP_SRM_NEVER_SET, -EIO,
	"STATUS_GRAPHICS_OPM_HDCP_SRM_NEVER_SET"},
	{STATUS_GRAPHICS_OPM_RESOLUTION_TOO_HIGH, -EIO,
	"STATUS_GRAPHICS_OPM_RESOLUTION_TOO_HIGH"},
	{STATUS_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE, -EIO,
	"STATUS_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE"},
	{STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_NO_LONGER_EXISTS, -EIO,
	"STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_NO_LONGER_EXISTS"},
	{STATUS_GRAPHICS_OPM_SESSION_TYPE_CHANGE_IN_PROGRESS, -EIO,
	"STATUS_GRAPHICS_OPM_SESSION_TYPE_CHANGE_IN_PROGRESS"},
	{STATUS_GRAPHICS_I2C_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_I2C_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST, -EIO,
	"STATUS_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST"},
	{STATUS_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA, -EIO,
	"STATUS_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA"},
	{STATUS_GRAPHICS_I2C_ERROR_RECEIVING_DATA, -EIO,
	"STATUS_GRAPHICS_I2C_ERROR_RECEIVING_DATA"},
	{STATUS_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_DDCCI_INVALID_DATA, -EIO,
	"STATUS_GRAPHICS_DDCCI_INVALID_DATA"},
	{STATUS_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE,
	-EIO,
	"STATUS_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE"},
	{STATUS_GRAPHICS_DDCCI_INVALID_CAPABILITIES_STRING, -EIO,
	"STATUS_GRAPHICS_DDCCI_INVALID_CAPABILITIES_STRING"},
	{STATUS_GRAPHICS_MCA_INTERNAL_ERROR, -EIO,
	"STATUS_GRAPHICS_MCA_INTERNAL_ERROR"},
	{STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND, -EIO,
	"STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND"},
	{STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH, -EIO,
	"STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH"},
	{STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM, -EIO,
	"STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM"},
	{STATUS_GRAPHICS_INVALID_PHYSICAL_MONITOR_HANDLE, -EIO,
	"STATUS_GRAPHICS_INVALID_PHYSICAL_MONITOR_HANDLE"},
	{STATUS_GRAPHICS_MONITOR_NO_LONGER_EXISTS, -EIO,
	"STATUS_GRAPHICS_MONITOR_NO_LONGER_EXISTS"},
	{STATUS_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED"},
	{STATUS_GRAPHICS_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME, -EIO,
	"STATUS_GRAPHICS_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME"},
	{STATUS_GRAPHICS_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP, -EIO,
	"STATUS_GRAPHICS_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP"},
	{STATUS_GRAPHICS_MIRRORING_DEVICES_NOT_SUPPORTED, -EIO,
	"STATUS_GRAPHICS_MIRRORING_DEVICES_NOT_SUPPORTED"},
	{STATUS_GRAPHICS_INVALID_POINTER, -EIO,
	"STATUS_GRAPHICS_INVALID_POINTER"},
	{STATUS_GRAPHICS_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE, -EIO,
	"STATUS_GRAPHICS_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE"},
	{STATUS_GRAPHICS_PARAMETER_ARRAY_TOO_SMALL, -EIO,
	"STATUS_GRAPHICS_PARAMETER_ARRAY_TOO_SMALL"},
	{STATUS_GRAPHICS_INTERNAL_ERROR, -EIO,
	"STATUS_GRAPHICS_INTERNAL_ERROR"},
	{STATUS_GRAPHICS_SESSION_TYPE_CHANGE_IN_PROGRESS, -EIO,
	"STATUS_GRAPHICS_SESSION_TYPE_CHANGE_IN_PROGRESS"},
	{STATUS_FVE_LOCKED_VOLUME, -EIO, "STATUS_FVE_LOCKED_VOLUME"},
	{STATUS_FVE_NOT_ENCRYPTED, -EIO, "STATUS_FVE_NOT_ENCRYPTED"},
	{STATUS_FVE_BAD_INFORMATION, -EIO, "STATUS_FVE_BAD_INFORMATION"},
	{STATUS_FVE_TOO_SMALL, -EIO, "STATUS_FVE_TOO_SMALL"},
	{STATUS_FVE_FAILED_WRONG_FS, -EIO, "STATUS_FVE_FAILED_WRONG_FS"},
	{STATUS_FVE_FAILED_BAD_FS, -EIO, "STATUS_FVE_FAILED_BAD_FS"},
	{STATUS_FVE_FS_NOT_EXTENDED, -EIO, "STATUS_FVE_FS_NOT_EXTENDED"},
	{STATUS_FVE_FS_MOUNTED, -EIO, "STATUS_FVE_FS_MOUNTED"},
	{STATUS_FVE_NO_LICENSE, -EIO, "STATUS_FVE_NO_LICENSE"},
	{STATUS_FVE_ACTION_NOT_ALLOWED, -EIO, "STATUS_FVE_ACTION_NOT_ALLOWED"},
	{STATUS_FVE_BAD_DATA, -EIO, "STATUS_FVE_BAD_DATA"},
	{STATUS_FVE_VOLUME_NOT_BOUND, -EIO, "STATUS_FVE_VOLUME_NOT_BOUND"},
	{STATUS_FVE_NOT_DATA_VOLUME, -EIO, "STATUS_FVE_NOT_DATA_VOLUME"},
	{STATUS_FVE_CONV_READ_ERROR, -EIO, "STATUS_FVE_CONV_READ_ERROR"},
	{STATUS_FVE_CONV_WRITE_ERROR, -EIO, "STATUS_FVE_CONV_WRITE_ERROR"},
	{STATUS_FVE_OVERLAPPED_UPDATE, -EIO, "STATUS_FVE_OVERLAPPED_UPDATE"},
	{STATUS_FVE_FAILED_SECTOR_SIZE, -EIO, "STATUS_FVE_FAILED_SECTOR_SIZE"},
	{STATUS_FVE_FAILED_AUTHENTICATION, -EIO,
	"STATUS_FVE_FAILED_AUTHENTICATION"},
	{STATUS_FVE_NOT_OS_VOLUME, -EIO, "STATUS_FVE_NOT_OS_VOLUME"},
	{STATUS_FVE_KEYFILE_NOT_FOUND, -EIO, "STATUS_FVE_KEYFILE_NOT_FOUND"},
	{STATUS_FVE_KEYFILE_INVALID, -EIO, "STATUS_FVE_KEYFILE_INVALID"},
	{STATUS_FVE_KEYFILE_NO_VMK, -EIO, "STATUS_FVE_KEYFILE_NO_VMK"},
	{STATUS_FVE_TPM_DISABLED, -EIO, "STATUS_FVE_TPM_DISABLED"},
	{STATUS_FVE_TPM_SRK_AUTH_NOT_ZERO, -EIO,
	"STATUS_FVE_TPM_SRK_AUTH_NOT_ZERO"},
	{STATUS_FVE_TPM_INVALID_PCR, -EIO, "STATUS_FVE_TPM_INVALID_PCR"},
	{STATUS_FVE_TPM_NO_VMK, -EIO, "STATUS_FVE_TPM_NO_VMK"},
	{STATUS_FVE_PIN_INVALID, -EIO, "STATUS_FVE_PIN_INVALID"},
	{STATUS_FVE_AUTH_INVALID_APPLICATION, -EIO,
	"STATUS_FVE_AUTH_INVALID_APPLICATION"},
	{STATUS_FVE_AUTH_INVALID_CONFIG, -EIO,
	"STATUS_FVE_AUTH_INVALID_CONFIG"},
	{STATUS_FVE_DEBUGGER_ENABLED, -EIO, "STATUS_FVE_DEBUGGER_ENABLED"},
	{STATUS_FVE_DRY_RUN_FAILED, -EIO, "STATUS_FVE_DRY_RUN_FAILED"},
	{STATUS_FVE_BAD_METADATA_POINTER, -EIO,
	"STATUS_FVE_BAD_METADATA_POINTER"},
	{STATUS_FVE_OLD_METADATA_COPY, -EIO, "STATUS_FVE_OLD_METADATA_COPY"},
	{STATUS_FVE_REBOOT_REQUIRED, -EIO, "STATUS_FVE_REBOOT_REQUIRED"},
	{STATUS_FVE_RAW_ACCESS, -EIO, "STATUS_FVE_RAW_ACCESS"},
	{STATUS_FVE_RAW_BLOCKED, -EIO, "STATUS_FVE_RAW_BLOCKED"},
	{STATUS_FWP_CALLOUT_NOT_FOUND, -EIO, "STATUS_FWP_CALLOUT_NOT_FOUND"},
	{STATUS_FWP_CONDITION_NOT_FOUND, -EIO,
	"STATUS_FWP_CONDITION_NOT_FOUND"},
	{STATUS_FWP_FILTER_NOT_FOUND, -EIO, "STATUS_FWP_FILTER_NOT_FOUND"},
	{STATUS_FWP_LAYER_NOT_FOUND, -EIO, "STATUS_FWP_LAYER_NOT_FOUND"},
	{STATUS_FWP_PROVIDER_NOT_FOUND, -EIO, "STATUS_FWP_PROVIDER_NOT_FOUND"},
	{STATUS_FWP_PROVIDER_CONTEXT_NOT_FOUND, -EIO,
	"STATUS_FWP_PROVIDER_CONTEXT_NOT_FOUND"},
	{STATUS_FWP_SUBLAYER_NOT_FOUND, -EIO, "STATUS_FWP_SUBLAYER_NOT_FOUND"},
	{STATUS_FWP_NOT_FOUND, -EIO, "STATUS_FWP_NOT_FOUND"},
	{STATUS_FWP_ALREADY_EXISTS, -EIO, "STATUS_FWP_ALREADY_EXISTS"},
	{STATUS_FWP_IN_USE, -EIO, "STATUS_FWP_IN_USE"},
	{STATUS_FWP_DYNAMIC_SESSION_IN_PROGRESS, -EIO,
	"STATUS_FWP_DYNAMIC_SESSION_IN_PROGRESS"},
	{STATUS_FWP_WRONG_SESSION, -EIO, "STATUS_FWP_WRONG_SESSION"},
	{STATUS_FWP_NO_TXN_IN_PROGRESS, -EIO, "STATUS_FWP_NO_TXN_IN_PROGRESS"},
	{STATUS_FWP_TXN_IN_PROGRESS, -EIO, "STATUS_FWP_TXN_IN_PROGRESS"},
	{STATUS_FWP_TXN_ABORTED, -EIO, "STATUS_FWP_TXN_ABORTED"},
	{STATUS_FWP_SESSION_ABORTED, -EIO, "STATUS_FWP_SESSION_ABORTED"},
	{STATUS_FWP_INCOMPATIBLE_TXN, -EIO, "STATUS_FWP_INCOMPATIBLE_TXN"},
	{STATUS_FWP_TIMEOUT, -ETIMEDOUT, "STATUS_FWP_TIMEOUT"},
	{STATUS_FWP_NET_EVENTS_DISABLED, -EIO,
	"STATUS_FWP_NET_EVENTS_DISABLED"},
	{STATUS_FWP_INCOMPATIBLE_LAYER, -EIO, "STATUS_FWP_INCOMPATIBLE_LAYER"},
	{STATUS_FWP_KM_CLIENTS_ONLY, -EIO, "STATUS_FWP_KM_CLIENTS_ONLY"},
	{STATUS_FWP_LIFETIME_MISMATCH, -EIO, "STATUS_FWP_LIFETIME_MISMATCH"},
	{STATUS_FWP_BUILTIN_OBJECT, -EIO, "STATUS_FWP_BUILTIN_OBJECT"},
	{STATUS_FWP_TOO_MANY_BOOTTIME_FILTERS, -EIO,
	"STATUS_FWP_TOO_MANY_BOOTTIME_FILTERS"},
	{STATUS_FWP_TOO_MANY_CALLOUTS, -EIO, "STATUS_FWP_TOO_MANY_CALLOUTS"},
	{STATUS_FWP_NOTIFICATION_DROPPED, -EIO,
	"STATUS_FWP_NOTIFICATION_DROPPED"},
	{STATUS_FWP_TRAFFIC_MISMATCH, -EIO, "STATUS_FWP_TRAFFIC_MISMATCH"},
	{STATUS_FWP_INCOMPATIBLE_SA_STATE, -EIO,
	"STATUS_FWP_INCOMPATIBLE_SA_STATE"},
	{STATUS_FWP_NULL_POINTER, -EIO, "STATUS_FWP_NULL_POINTER"},
	{STATUS_FWP_INVALID_ENUMERATOR, -EIO, "STATUS_FWP_INVALID_ENUMERATOR"},
	{STATUS_FWP_INVALID_FLAGS, -EIO, "STATUS_FWP_INVALID_FLAGS"},
	{STATUS_FWP_INVALID_NET_MASK, -EIO, "STATUS_FWP_INVALID_NET_MASK"},
	{STATUS_FWP_INVALID_RANGE, -EIO, "STATUS_FWP_INVALID_RANGE"},
	{STATUS_FWP_INVALID_INTERVAL, -EIO, "STATUS_FWP_INVALID_INTERVAL"},
	{STATUS_FWP_ZERO_LENGTH_ARRAY, -EIO, "STATUS_FWP_ZERO_LENGTH_ARRAY"},
	{STATUS_FWP_NULL_DISPLAY_NAME, -EIO, "STATUS_FWP_NULL_DISPLAY_NAME"},
	{STATUS_FWP_INVALID_ACTION_TYPE, -EIO,
	"STATUS_FWP_INVALID_ACTION_TYPE"},
	{STATUS_FWP_INVALID_WEIGHT, -EIO, "STATUS_FWP_INVALID_WEIGHT"},
	{STATUS_FWP_MATCH_TYPE_MISMATCH, -EIO,
	"STATUS_FWP_MATCH_TYPE_MISMATCH"},
	{STATUS_FWP_TYPE_MISMATCH, -EIO, "STATUS_FWP_TYPE_MISMATCH"},
	{STATUS_FWP_OUT_OF_BOUNDS, -EIO, "STATUS_FWP_OUT_OF_BOUNDS"},
	{STATUS_FWP_RESERVED, -EIO, "STATUS_FWP_RESERVED"},
	{STATUS_FWP_DUPLICATE_CONDITION, -EIO,
	"STATUS_FWP_DUPLICATE_CONDITION"},
	{STATUS_FWP_DUPLICATE_KEYMOD, -EIO, "STATUS_FWP_DUPLICATE_KEYMOD"},
	{STATUS_FWP_ACTION_INCOMPATIBLE_WITH_LAYER, -EIO,
	"STATUS_FWP_ACTION_INCOMPATIBLE_WITH_LAYER"},
	{STATUS_FWP_ACTION_INCOMPATIBLE_WITH_SUBLAYER, -EIO,
	"STATUS_FWP_ACTION_INCOMPATIBLE_WITH_SUBLAYER"},
	{STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_LAYER, -EIO,
	"STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_LAYER"},
	{STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_CALLOUT, -EIO,
	"STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_CALLOUT"},
	{STATUS_FWP_INCOMPATIBLE_AUTH_METHOD, -EIO,
	"STATUS_FWP_INCOMPATIBLE_AUTH_METHOD"},
	{STATUS_FWP_INCOMPATIBLE_DH_GROUP, -EIO,
	"STATUS_FWP_INCOMPATIBLE_DH_GROUP"},
	{STATUS_FWP_EM_NOT_SUPPORTED, -EOPNOTSUPP,
	"STATUS_FWP_EM_NOT_SUPPORTED"},
	{STATUS_FWP_NEVER_MATCH, -EIO, "STATUS_FWP_NEVER_MATCH"},
	{STATUS_FWP_PROVIDER_CONTEXT_MISMATCH, -EIO,
	"STATUS_FWP_PROVIDER_CONTEXT_MISMATCH"},
	{STATUS_FWP_INVALID_PARAMETER, -EIO, "STATUS_FWP_INVALID_PARAMETER"},
	{STATUS_FWP_TOO_MANY_SUBLAYERS, -EIO, "STATUS_FWP_TOO_MANY_SUBLAYERS"},
	{STATUS_FWP_CALLOUT_NOTIFICATION_FAILED, -EIO,
	"STATUS_FWP_CALLOUT_NOTIFICATION_FAILED"},
	{STATUS_FWP_INCOMPATIBLE_AUTH_CONFIG, -EIO,
	"STATUS_FWP_INCOMPATIBLE_AUTH_CONFIG"},
	{STATUS_FWP_INCOMPATIBLE_CIPHER_CONFIG, -EIO,
	"STATUS_FWP_INCOMPATIBLE_CIPHER_CONFIG"},
	{STATUS_FWP_TCPIP_NOT_READY, -EIO, "STATUS_FWP_TCPIP_NOT_READY"},
	{STATUS_FWP_INJECT_HANDLE_CLOSING, -EIO,
	"STATUS_FWP_INJECT_HANDLE_CLOSING"},
	{STATUS_FWP_INJECT_HANDLE_STALE, -EIO,
	"STATUS_FWP_INJECT_HANDLE_STALE"},
	{STATUS_FWP_CANNOT_PEND, -EIO, "STATUS_FWP_CANNOT_PEND"},
	{STATUS_NDIS_CLOSING, -EIO, "STATUS_NDIS_CLOSING"},
	{STATUS_NDIS_BAD_VERSION, -EIO, "STATUS_NDIS_BAD_VERSION"},
	{STATUS_NDIS_BAD_CHARACTERISTICS, -EIO,
	"STATUS_NDIS_BAD_CHARACTERISTICS"},
	{STATUS_NDIS_ADAPTER_NOT_FOUND, -EIO, "STATUS_NDIS_ADAPTER_NOT_FOUND"},
	{STATUS_NDIS_OPEN_FAILED, -EIO, "STATUS_NDIS_OPEN_FAILED"},
	{STATUS_NDIS_DEVICE_FAILED, -EIO, "STATUS_NDIS_DEVICE_FAILED"},
	{STATUS_NDIS_MULTICAST_FULL, -EIO, "STATUS_NDIS_MULTICAST_FULL"},
	{STATUS_NDIS_MULTICAST_EXISTS, -EIO, "STATUS_NDIS_MULTICAST_EXISTS"},
	{STATUS_NDIS_MULTICAST_NOT_FOUND, -EIO,
	"STATUS_NDIS_MULTICAST_NOT_FOUND"},
	{STATUS_NDIS_REQUEST_ABORTED, -EIO, "STATUS_NDIS_REQUEST_ABORTED"},
	{STATUS_NDIS_RESET_IN_PROGRESS, -EIO, "STATUS_NDIS_RESET_IN_PROGRESS"},
	{STATUS_NDIS_INVALID_PACKET, -EIO, "STATUS_NDIS_INVALID_PACKET"},
	{STATUS_NDIS_INVALID_DEVICE_REQUEST, -EIO,
	"STATUS_NDIS_INVALID_DEVICE_REQUEST"},
	{STATUS_NDIS_ADAPTER_NOT_READY, -EIO, "STATUS_NDIS_ADAPTER_NOT_READY"},
	{STATUS_NDIS_INVALID_LENGTH, -EIO, "STATUS_NDIS_INVALID_LENGTH"},
	{STATUS_NDIS_INVALID_DATA, -EIO, "STATUS_NDIS_INVALID_DATA"},
	{STATUS_NDIS_BUFFER_TOO_SHORT, -ENOBUFS,
	"STATUS_NDIS_BUFFER_TOO_SHORT"},
	{STATUS_NDIS_INVALID_OID, -EIO, "STATUS_NDIS_INVALID_OID"},
	{STATUS_NDIS_ADAPTER_REMOVED, -EIO, "STATUS_NDIS_ADAPTER_REMOVED"},
	{STATUS_NDIS_UNSUPPORTED_MEDIA, -EIO, "STATUS_NDIS_UNSUPPORTED_MEDIA"},
	{STATUS_NDIS_GROUP_ADDRESS_IN_USE, -EIO,
	"STATUS_NDIS_GROUP_ADDRESS_IN_USE"},
	{STATUS_NDIS_FILE_NOT_FOUND, -EIO, "STATUS_NDIS_FILE_NOT_FOUND"},
	{STATUS_NDIS_ERROR_READING_FILE, -EIO,
	"STATUS_NDIS_ERROR_READING_FILE"},
	{STATUS_NDIS_ALREADY_MAPPED, -EIO, "STATUS_NDIS_ALREADY_MAPPED"},
	{STATUS_NDIS_RESOURCE_CONFLICT, -EIO, "STATUS_NDIS_RESOURCE_CONFLICT"},
	{STATUS_NDIS_MEDIA_DISCONNECTED, -EIO,
	"STATUS_NDIS_MEDIA_DISCONNECTED"},
	{STATUS_NDIS_INVALID_ADDRESS, -EIO, "STATUS_NDIS_INVALID_ADDRESS"},
	{STATUS_NDIS_PAUSED, -EIO, "STATUS_NDIS_PAUSED"},
	{STATUS_NDIS_INTERFACE_NOT_FOUND, -EIO,
	"STATUS_NDIS_INTERFACE_NOT_FOUND"},
	{STATUS_NDIS_UNSUPPORTED_REVISION, -EIO,
	"STATUS_NDIS_UNSUPPORTED_REVISION"},
	{STATUS_NDIS_INVALID_PORT, -EIO, "STATUS_NDIS_INVALID_PORT"},
	{STATUS_NDIS_INVALID_PORT_STATE, -EIO,
	"STATUS_NDIS_INVALID_PORT_STATE"},
	{STATUS_NDIS_LOW_POWER_STATE, -EIO, "STATUS_NDIS_LOW_POWER_STATE"},
	{STATUS_NDIS_NOT_SUPPORTED, -ENOSYS, "STATUS_NDIS_NOT_SUPPORTED"},
	{STATUS_NDIS_DOT11_AUTO_CONFIG_ENABLED, -EIO,
	"STATUS_NDIS_DOT11_AUTO_CONFIG_ENABLED"},
	{STATUS_NDIS_DOT11_MEDIA_IN_USE, -EIO,
	"STATUS_NDIS_DOT11_MEDIA_IN_USE"},
	{STATUS_NDIS_DOT11_POWER_STATE_INVALID, -EIO,
	"STATUS_NDIS_DOT11_POWER_STATE_INVALID"},
	{STATUS_IPSEC_BAD_SPI, -EIO, "STATUS_IPSEC_BAD_SPI"},
	{STATUS_IPSEC_SA_LIFETIME_EXPIRED, -EIO,
	"STATUS_IPSEC_SA_LIFETIME_EXPIRED"},
	{STATUS_IPSEC_WRONG_SA, -EIO, "STATUS_IPSEC_WRONG_SA"},
	{STATUS_IPSEC_REPLAY_CHECK_FAILED, -EIO,
	"STATUS_IPSEC_REPLAY_CHECK_FAILED"},
	{STATUS_IPSEC_INVALID_PACKET, -EIO, "STATUS_IPSEC_INVALID_PACKET"},
	{STATUS_IPSEC_INTEGRITY_CHECK_FAILED, -EIO,
	"STATUS_IPSEC_INTEGRITY_CHECK_FAILED"},
	{STATUS_IPSEC_CLEAR_TEXT_DROP, -EIO, "STATUS_IPSEC_CLEAR_TEXT_DROP"},
	{0, 0, NULL}
};

/*****************************************************************************
 Print an error message from the status code
 *****************************************************************************/
static void
smb2_print_status(__le32 status)
{
	int idx = 0;

	while (smb2_error_map_table[idx].status_string != NULL) {
		if ((smb2_error_map_table[idx].smb2_status) == status) {
			pr_notice("Status code returned 0x%08x %s\n", status,
				  smb2_error_map_table[idx].status_string);
		}
		idx++;
	}
	return;
}

int
map_smb2_to_linux_error(char *buf, bool log_err)
{
	struct smb2_sync_hdr *shdr = get_sync_hdr(buf);
	unsigned int i;
	int rc = -EIO;
	__le32 smb2err = shdr->Status;

	if (smb2err == 0)
		return 0;

	/* mask facility */
	if (log_err && (smb2err != STATUS_MORE_PROCESSING_REQUIRED) &&
	    (smb2err != STATUS_END_OF_FILE))
		smb2_print_status(smb2err);
	else if (cifsFYI & CIFS_RC)
		smb2_print_status(smb2err);

	for (i = 0; i < sizeof(smb2_error_map_table) /
			sizeof(struct status_to_posix_error); i++) {
		if (smb2_error_map_table[i].smb2_status == smb2err) {
			rc = smb2_error_map_table[i].posix_error;
			break;
		}
	}

	/* on error mapping not found  - return EIO */

	cifs_dbg(FYI, "Mapping SMB2 status code 0x%08x to POSIX err %d\n",
		 __le32_to_cpu(smb2err), rc);

	return rc;
}
